package com.agilex.healthcare.mobilehealthplatform.datalayer.contactlog;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.Date;
import java.util.UUID;

import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.ContactLog;
import com.agilex.healthcare.mobilehealthplatform.domain.ContactLogs;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.utility.DateHelper;

public class ContactLogDataLayerExcelTest {
	
	ContactLogDataLayer dl = new ContactLogDataLayerExcel();

	@Test
	public void getGallowContactLogs() {
		String gallowPatientId = "D123401";
		ContactLogDataLayer dl = new ContactLogDataLayerExcel();
		ContactLogs contactLogs = dl.getContactLogs(gallowPatientId);
		assertTrue(contactLogs.size() > 0);
		
		ContactLog contactLog01 = contactLogs.get(0);
		assertEquals("1", contactLog01.getUniqueId());

		assertEquals(DateHelper.parseDate("3/1/2012"), contactLog01.getDate());
	}

	@Test
	public void getAldieContactLogs() {
		String gallowPatientId = "PATID26";
		ContactLogDataLayer dl = new ContactLogDataLayerExcel();
		ContactLogs contactLogs = dl.getContactLogs(gallowPatientId);
		assertTrue(contactLogs.size() > 0);
		
		ContactLog contactLog01 = contactLogs.get(0);
		assertEquals("1", contactLog01.getUniqueId());

		assertEquals(DateHelper.parseDate("3/1/2012"), contactLog01.getDate());
	}
	
	@Test
	public void get0ResultsForInvalidPatient() {
		String patientId = "invalid";
		ContactLogs contactLogs = dl.getContactLogs(patientId);
		assertEquals(0, contactLogs.size());
	}
	
	@Test
	public void addAndRetrieveContactLogs() {
		String patientId = generateId();
		ContactLogs contactLogs = dl.getContactLogs(patientId);
		assertEquals(0, contactLogs.size());
		
		ContactLog contactLog = new ContactLog();
		contactLog.setPatientId(patientId);
		assertEquals(patientId,contactLog.getPatientId());
		ContactLog savedContactLog = dl.saveContactLog(contactLog);
		
		ContactLogs retrieved = dl.getContactLogs(patientId);
		assertEquals(1, retrieved.size());
		assertEquals(savedContactLog.getUniqueId(), retrieved.get(0).getUniqueId());
	}

	@Test
	public void retrieveWithDateFilter() {
		String patientId = generateId();
		
		ContactLog contactLog = new ContactLog();
		contactLog.setPatientId(patientId);
		contactLog.setLogTime(DateHelper.parseDate("1/12/2012"));
		dl.saveContactLog(contactLog);
		
		contactLog = new ContactLog();
		contactLog.setPatientId(patientId);
		contactLog.setLogTime(DateHelper.parseDate("1/13/2012"));
		dl.saveContactLog(contactLog);
		
		contactLog = new ContactLog();
		contactLog.setPatientId(patientId);
		contactLog.setLogTime(DateHelper.parseDate("1/15/2012"));
		dl.saveContactLog(contactLog);
		
		Date startDate = DateHelper.parseDate("1/12/2012");
		Date endDate = DateHelper.parseDate("1/14/2012");
		DateFilter filter = DateFilterFactory.createFilterFromDate(startDate, endDate);
		ContactLogs shouldBeTwo = dl.getContactLogs(patientId, filter);
		assertEquals(2, shouldBeTwo.size());
		
		startDate = DateHelper.parseDate("1/12/2012");
		endDate = DateHelper.parseDate("1/15/2012");
		filter = DateFilterFactory.createFilterFromDate(startDate, endDate);
		ContactLogs shouldBeThree = dl.getContactLogs(patientId, filter);
		assertEquals(3, shouldBeThree.size());
		
		startDate = DateHelper.parseDate("1/12/2012");
		endDate = DateHelper.parseDate("1/12/2012");
		filter = DateFilterFactory.createFilterFromDate(startDate, endDate);
		ContactLogs shouldBeOne = dl.getContactLogs(patientId, filter);
		assertEquals(1, shouldBeOne.size());
	}
	
	@Test 
	public void validateUpdate() {
		String patientId = generateId();
		
		ContactLog contactLog = new ContactLog();
		contactLog.setPatientId(patientId);
		contactLog.setActivity("1st");
		contactLog.setLogTime(DateHelper.parseDate("1/12/2012"));
		ContactLog saved = dl.saveContactLog(contactLog);
		assertNotNull(saved);
		assertEquals("1st", saved.getActivity());
		
		saved.setActivity("2nd");
		ContactLog updated = dl.saveContactLog(saved);
		
		assertEquals("2nd", updated.getActivity());
	}
	
	@Test 
	public void validateDelete() {
		String patientId = generateId();
		
		ContactLog contactLog = new ContactLog();
		contactLog.setPatientId(patientId);
		contactLog.setLogTime(DateHelper.parseDate("1/12/2012"));
		dl.saveContactLog(contactLog);
		
		ContactLogs shouldBeOne = dl.getContactLogs(patientId);
		assertEquals(1, shouldBeOne.size());
		
		dl.deleteContactLog(shouldBeOne.get(0));
		
		ContactLogs shouldBeEmpty = dl.getContactLogs(patientId);
		assertEquals(0, shouldBeEmpty.size());
	}
	
	private String generateId() {
		return UUID.randomUUID().toString();
	}

	@Test
	public void testActivity() {
		String gallowPatientId = "D123401";
		ContactLogDataLayer dl = new ContactLogDataLayerExcel();
		ContactLogs contactLogs = dl.getContactLogs(gallowPatientId);
		assertTrue(contactLogs.size() > 0);
		ContactLog contactLog01 = contactLogs.get(0);
		assertEquals("Phone Call", contactLog01.getActivity());
	}

	@Test
	public void testContact() {
		String gallowPatientId = "D123401";
		ContactLogDataLayer dl = new ContactLogDataLayerExcel();
		ContactLogs contactLogs = dl.getContactLogs(gallowPatientId);
		assertTrue(contactLogs.size() > 0);
		ContactLog contactLog01 = contactLogs.get(0);
		assertEquals("Dr. Fitness", contactLog01.getContact());		
	}

	@Test
	public void testLogTime() {
		String gallowPatientId = "D123401";
		ContactLogDataLayer dl = new ContactLogDataLayerExcel();
		ContactLogs contactLogs = dl.getContactLogs(gallowPatientId);
		assertTrue(contactLogs.size() > 0);
		ContactLog contactLog01 = contactLogs.get(0);
		assertEquals(DateHelper.parseDate("3/1/2012"), contactLog01.getLogTime());
	}

	@Test
	public void testNote() {
		String gallowPatientId = "D123401";
		ContactLogDataLayer dl = new ContactLogDataLayerExcel();
		ContactLogs contactLogs = dl.getContactLogs(gallowPatientId);
		assertTrue(contactLogs.size() > 0);
		ContactLog contactLog01 = contactLogs.get(0);
		assertEquals("Be More Active", contactLog01.getNote());
	}

	@Test
	public void testTask() {
		String gallowPatientId = "D123401";
		ContactLogDataLayer dl = new ContactLogDataLayerExcel();
		ContactLogs contactLogs = dl.getContactLogs(gallowPatientId);
		assertTrue(contactLogs.size() > 0);
		ContactLog contactLog01 = contactLogs.get(0);
		assertEquals("Morning Jog", contactLog01.getTask());		
	}

	@Test
	public void testSourceSystem() {
		
	}
}
