package com.agilex.healthcare.mobilehealthplatform.datalayer.audit;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.Date;
import java.util.UUID;

import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.AuditLog;
import com.agilex.healthcare.mobilehealthplatform.domain.AuditLogs;
import com.agilex.healthcare.mobilehealthplatform.domain.code.AuditLogCode;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.utility.DateHelper;

public class AuditLogDataLayerStubTest {
	private AuditLogDataLayer dataLayer = new AuditLogDataLayerStub();
	private String testUserId = UUID.randomUUID().toString();
	private String testUserId2 = UUID.randomUUID().toString();
	
	@Test
	public void verifySaveAuditLog() {
		AuditLog auditLog = new AuditLog(testUserId, AuditLogCode.NOTIFICATION, AuditLogCode.NOTIFICATION_SUBTYPE_SENT, new Date(), "Details");
		AuditLog savedAuditLog = dataLayer.saveAudit(auditLog);

		assertNotNull(savedAuditLog);
		assertNotNull(savedAuditLog.getId());

		AuditLogFilter emptyFilter = new AuditLogFilter();
		AuditLogs retrievedAuditLogs = dataLayer.fetchAuditLogs(testUserId, emptyFilter);

		assertNotNull(retrievedAuditLogs);
		assertEquals(1, retrievedAuditLogs.size());		
	}
	
	@Test
	public void verifyFetchAuditLogs() {
		generateAuditLogs(testUserId, 10);
		
		AuditLogFilter emptyFilter = new AuditLogFilter();
		AuditLogs auditLogs = dataLayer.fetchAuditLogs(testUserId, emptyFilter);

		assertNotNull(auditLogs);
		assertEquals(10, auditLogs.size());
	}
	
	@Test
	public void verifyFetchAuditLog() {
		String userId = UUID.randomUUID().toString();
		AuditLog auditLog = new AuditLog(userId, AuditLogCode.NOTIFICATION, AuditLogCode.NOTIFICATION_SUBTYPE_SENT, new Date(), "");
		
		AuditLog savedAuditLog = dataLayer.saveAudit(auditLog);
		AuditLog retrievedAuditLog = dataLayer.fetchAuditLog(userId, savedAuditLog.getId());
		assertEquals(savedAuditLog.getUserId(), retrievedAuditLog.getUserId());
		assertEquals(savedAuditLog.getDetails(), retrievedAuditLog.getDetails());
		assertEquals(savedAuditLog.getType(), retrievedAuditLog.getType());
		assertEquals(savedAuditLog.getSubType(), retrievedAuditLog.getSubType());
	}

	@Test
	public void verifyGetAuditLogsIsEmptyForNewUser() {
		generateAuditLogs(testUserId, 10);
		
		AuditLogFilter emptyFilter = new AuditLogFilter();
		AuditLogs auditLogs = dataLayer.fetchAuditLogs("non-existant-user", emptyFilter);

		assertEquals(0, auditLogs.size());
	}
	
	@Test
	public void verifyGetAuditLogsByUserId() {
		generateAuditLogs(testUserId, 10);
		generateAuditLogs(testUserId2, 10);

		AuditLogFilter emptyFilter = new AuditLogFilter();
		AuditLogs auditLogs = dataLayer.fetchAuditLogs(testUserId2, emptyFilter);

		assertNotNull(auditLogs);
		assertTrue(auditLogs.size() == 10);
	}

	@Test
	public void verifyFetchAuditLogsByDate() {
		generateAuditLogs(testUserId, new Date(), 10);
		generateAuditLogs(testUserId, DateHelper.getTomorrow(), 10);

		DateFilter yesterdayFilter = DateFilterFactory.createFilterFromToday();
		AuditLogFilter auditLogFilterByDate = new AuditLogFilter(yesterdayFilter);
		AuditLogs auditLogs = dataLayer.fetchAuditLogs(testUserId, auditLogFilterByDate);

		assertNotNull(auditLogs);
		assertEquals(10, auditLogs.size());
	}
	
	@Test
	public void verifyFetchAuditLogsByType() {
		generateAuditLogs(testUserId, AuditLogCode.NOTIFICATION, 10);
		generateAuditLogs(testUserId, "other type", 10);

		AuditLogFilter auditLogFilterByType = new AuditLogFilter();
		auditLogFilterByType.setType(AuditLogCode.NOTIFICATION);
		AuditLogs auditLogs = dataLayer.fetchAuditLogs(testUserId, auditLogFilterByType);

		assertNotNull(auditLogs);
		assertEquals(10, auditLogs.size());
	}

	@Test
	public void verifyFetchAuditLogsBySubType() {
		generateAuditLogs(testUserId, AuditLogCode.NOTIFICATION, AuditLogCode.NOTIFICATION_SUBTYPE_RETRIEVED, 10);
		generateAuditLogs(testUserId, "other type", "other subtype", 10);

		AuditLogFilter auditLogFilterBySubType = new AuditLogFilter();
		auditLogFilterBySubType.setSubType(AuditLogCode.NOTIFICATION_SUBTYPE_RETRIEVED);
		AuditLogs auditLogs = dataLayer.fetchAuditLogs(testUserId, auditLogFilterBySubType);

		assertNotNull(auditLogs);
		assertEquals(10, auditLogs.size());
	}
	
	public void generateAuditLogs(String userId, int numberOfRecords) {
		for(int i = 0; i < numberOfRecords; i++) {
			AuditLog log = new AuditLog(userId, "Type", "Subtype", new Date(), "Details "+i);
			dataLayer.saveAudit(log);
		}
	}

	public void generateAuditLogs(String userId, String auditLogType, int numberOfRecords) {
		for(int i = 0; i < numberOfRecords; i++) {
			AuditLog log = new AuditLog(userId, auditLogType, "Subtype", new Date(), "Details "+i);
			dataLayer.saveAudit(log);
		}
	}

	public void generateAuditLogs(String userId, String auditLogType, String auditLogSubType, int numberOfRecords) {
		for(int i = 0; i < numberOfRecords; i++) {
			AuditLog log = new AuditLog(userId, auditLogType, auditLogSubType, new Date(), "Details "+i);
			dataLayer.saveAudit(log);
		}
	}

	public void generateAuditLogs(String userId, Date date, int numberOfRecords) {
		for(int i = 0; i < numberOfRecords; i++) {
			AuditLog log = new AuditLog(userId, "Type", "Subtype", date, "Details "+i);
			dataLayer.saveAudit(log);
		}
	}
}
