package com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.Date;
import java.util.UUID;

import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResult;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResultReport;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResults;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.ToolCategories;
import com.agilex.healthcare.mobilehealthplatform.domain.ToolCategory;
import com.agilex.healthcare.mobilehealthplatform.domain.code.AssessmentCode;
import com.agilex.healthcare.utility.DateHelper;

public class AssessmentResultsDataLayerMockTest {
	AssessmentResultsDataLayerMock datalayerMock = new AssessmentResultsDataLayerMock();

	Date dateTaken = new Date();
	AssessmentResult assessmentResult;

	@Test
	public void verifyFetchSpecificResult() {
		PatientIdentifier patientIdentifier = getRandomPatientIdentifier();
		assessmentResult = createAndSaveBasicAssessmentResult(patientIdentifier, dateTaken, 10, false);
		String resultId = assessmentResult.getUniqueId();
		AssessmentResult retrievedAssessmentResult = datalayerMock.fetchAssessmentResult(patientIdentifier, resultId);
		assertNotNull(retrievedAssessmentResult);
	}

	@Test
	public void verifySaveOfBasicAssessmentResult() {
		PatientIdentifier patientIdentifier = getRandomPatientIdentifier();
		assessmentResult = createAndSaveBasicAssessmentResult(patientIdentifier, dateTaken, 10, false);

		assertNotNull(assessmentResult);
		assertEquals("value for unique title", assessmentResult.getUniqueTitle());
	}

	@Test
	public void verifySaveOfNewResult() {
		PatientIdentifier patientIdentifier = getRandomPatientIdentifier();
		assessmentResult = createAndSaveBasicAssessmentResult(patientIdentifier, dateTaken, 10, false);
		assessmentResult = createAndSaveBasicAssessmentResult(patientIdentifier, dateTaken, 12, false);

		AssessmentResults savedAssessmentResults = datalayerMock.fetchAssessmentResultsByPatient(patientIdentifier.getUniqueId());
		assertNotNull(savedAssessmentResults);
		assertEquals(2, savedAssessmentResults.size());
	}

	@Test
	public void verifySaveOfDraft() {
		PatientIdentifier patientIdentifier = getRandomPatientIdentifier();
		assessmentResult = createAndSaveBasicAssessmentResult(patientIdentifier, dateTaken, 10, false);
		assessmentResult = createAndSaveBasicAssessmentResult(patientIdentifier, dateTaken, 12, true);

		AssessmentResults savedAssessmentResults = datalayerMock.fetchAssessmentDraftsByPatient(patientIdentifier);
		assertNotNull(savedAssessmentResults);
		assertEquals(1, savedAssessmentResults.size());
	}

	@Test
	public void verifySaveReport() {
		PatientIdentifier patientIdentifier = getRandomPatientIdentifier();
		assessmentResult = createAndSaveBasicAssessmentResult(patientIdentifier, dateTaken, 10, false);

		AssessmentResult savedAssessmentResult = datalayerMock.fetchMostRecentlyCompletedResult("testAssessmentId", patientIdentifier);

		assertEquals("my-tool", savedAssessmentResult.getAssessmentResultReport().getToolCategories().get(0).getName());
	}

	@Test
	public void verifyFetchPreviouslyCompletedResult() {
		PatientIdentifier patientIdentifier = getRandomPatientIdentifier();
		Date oldestDate = DateHelper.getPreviousMonth();
		Date middleDate = DateHelper.getYesterday();
		Date newestDate = DateHelper.getToday();

		assessmentResult = createAndSaveBasicAssessmentResult(patientIdentifier, oldestDate, 12, false);
		assessmentResult = createAndSaveBasicAssessmentResult(patientIdentifier, middleDate, 11, false);
		assessmentResult = createAndSaveBasicAssessmentResult(patientIdentifier, newestDate, 10, false);

		AssessmentResult mostRecentlyCompletedResult = datalayerMock.fetchMostRecentlyCompletedResult("testAssessmentId", patientIdentifier);
		assertTrue(oldestDate.before(newestDate));
		assertEquals(10, mostRecentlyCompletedResult.getScore());
	}

	private AssessmentResult createAndSaveBasicAssessmentResult(PatientIdentifier patientIdentifier, Date dateTaken, int score, boolean inProgress) {

		AssessmentResult assessmentResult = new AssessmentResult();
		assessmentResult.setPatientIdentifier(patientIdentifier);
		assessmentResult.setDateTaken(dateTaken);
		assessmentResult.setAssessmentId("testAssessmentId");
		assessmentResult.setUniqueTitle("value for unique title");
		assessmentResult.setUniqueId(getRandomUniqueId());
		assessmentResult.setVersion("value for version");
		assessmentResult.setAuthenticationStrategy("value for authentication strategy");
		assessmentResult.setNotes("value for notes");
		assessmentResult.setScore(score);
		assessmentResult.setScoringAlgorithm(AssessmentCode.SCORING_ALGORITHM_UNSCORED);
		assessmentResult.setInProgress(inProgress);
		assessmentResult.setWaitingPeriod(7);

		AssessmentResultReport assessmentResultReport = new AssessmentResultReport();
		ToolCategories toolCategories = new ToolCategories();
		ToolCategory tool = new ToolCategory("my-tool");
		toolCategories.add(tool);
		assessmentResultReport.setToolCategories(toolCategories);
		assessmentResult.setAssessmentResultReport(assessmentResultReport);

		datalayerMock.saveAssessmentResult(assessmentResult);
		return assessmentResult;
	}

	private PatientIdentifier getRandomPatientIdentifier() {
		return new PatientIdentifier("EDIPI", UUID.randomUUID().toString());
	}

	private String getRandomUniqueId() {
		return UUID.randomUUID().toString();
	}
}