package com.agilex.healthcare.mobilehealthplatform.datalayer.appointment;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.Date;

import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.Appointment;
import com.agilex.healthcare.mobilehealthplatform.domain.Appointments;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullChecker;

public class AppointmentsDataLayerExcelTest {

    private PatientIdentifier createMockPatientIdentifier(String uniqueId){
        return new PatientIdentifier("mock-aa", uniqueId);
    }

	@Test
	public void retrievePatientAppointmentNoResults() throws Exception {
		AppointmentDataLayer dao = getDataLayer();
		String patientId = "invalid";
		Appointments appointments = dao.fetchAppointmentsByPatient(createMockPatientIdentifier(patientId), DateFilterFactory.createEmptyFilter());
		assertEquals(0, appointments.size());
	}

	@Test
	public void retrievePatientAppointment() throws Exception {
		AppointmentDataLayer dao = getDataLayer();
		String patientId = "D123401";
		Appointments appointments = dao.fetchAppointmentsByPatient(createMockPatientIdentifier(patientId), DateFilterFactory.createEmptyFilter());
		assertTrue(appointments.size() >= 98);
	}

	@Test
	public void retrievePatientAppointmentWithDateFilter() throws Exception {
		AppointmentDataLayer dao = getDataLayer();
		String patientId = "D123401";
		Date beginDate = DateHelper.parseDate("1/2/2012");
		Date endDate = DateHelper.parseDate("1/3/2012");
		Appointments appointments = dao.fetchAppointmentsByPatient(createMockPatientIdentifier(patientId), DateFilterFactory.createFilterFromDate(beginDate, endDate));
		assertEquals(1, appointments.size());
	}

	@Test
	public void retrievePatientAppointmentWithDateFilterNoResults() throws Exception {
		AppointmentDataLayer dao = getDataLayer();
		String patientId = "D123401";
		Date beginDate = DateHelper.parseDate("1/2/2011");
		Date endDate = DateHelper.parseDate("1/3/2011");
		Appointments appointments = dao.fetchAppointmentsByPatient(createMockPatientIdentifier(patientId), DateFilterFactory.createFilterFromDate(beginDate, endDate));
		assertEquals(0, appointments.size());
	}

	@Test
	public void retrievePatientAppointmentWithBeginDateFilter() throws Exception {
		AppointmentDataLayer dao = getDataLayer();
		String patientId = "D123401";
		Date beginDate = DateHelper.parseDate("1/2/2012");
		Date endDate = null;
		Appointments appointments = dao.fetchAppointmentsByPatient(createMockPatientIdentifier(patientId), DateFilterFactory.createFilterFromDate(beginDate, endDate));
		assertTrue(appointments.size() >= 97);
	}

	@Test
	public void retrievePatientAppointmentWithEndDateFilter() throws Exception {
		AppointmentDataLayer dao = getDataLayer();
		String patientId = "D123401";
		Date beginDate = null;
		Date endDate = DateHelper.parseDate("1/3/2012");
		Appointments appointments = dao.fetchAppointmentsByPatient(createMockPatientIdentifier(patientId), DateFilterFactory.createFilterFromDate(beginDate, endDate));
		assertEquals(4, appointments.size());
	}

	@Test
	public void validateClinicPhone() throws Exception {
		AppointmentDataLayer dao = getDataLayer();
		String patientId = "D123401";
		Date beginDate = new Date();
		Date endDate = DateHelper.get30DaysFromNow();
		Appointments appointments = dao.fetchAppointmentsByPatient(createMockPatientIdentifier(patientId), DateFilterFactory.createFilterFromDate(beginDate, endDate));
		assertTrue(appointments.size() > 0);
		boolean isPassed = false;
		for (Appointment appointment : appointments) {
			if(appointment.getUniqueId().equalsIgnoreCase("101") && NullChecker.isNotNullish(appointment.getClinicPhone())){
				isPassed = true;
			}
		}
		assertTrue(isPassed);
	}
	
	private AppointmentDataLayer getDataLayer() {
		AppointmentDataLayer dao = new AppointmentsDataLayerExcel();
		return dao;
	}

	@Test
	public void retrieveByProvider() throws Exception {
		AppointmentDataLayer dao = getDataLayer();
		Appointments appointments = dao.getProviderAppointments("provider1", DateFilterFactory.createEmptyFilter());

		for (Appointment a : appointments) {
			System.out.println(String.format("appointment [id=%s][p=%s][dt=%s]", a.getUniqueId(), a.getPatientId(), a.getAppointmentStartDate()));
		}

		assertTrue(appointments.size() > 0);
	}

	@Test
	public void retrievePatientAppointmentListDoesNotContainNote() throws Exception {
		AppointmentDataLayer dao = getDataLayer();
		String patientId = "D123401";
		Appointments appointments = dao.fetchAppointmentsByPatient(createMockPatientIdentifier(patientId), DateFilterFactory.createEmptyFilter());
		assertTrue(appointments.size() > 0);
		for (Appointment appointment : appointments) {
			assertTrue(NullChecker.isNullish(appointment.getClinicalNotes()));
		}
	}

	@Test
	public void retrieveSingleAppointmentContainNote() throws Exception {
		AppointmentDataLayer dao = getDataLayer();
		String patientId = "D123401";
		Appointments appointments = dao.fetchAppointmentsByPatient(createMockPatientIdentifier(patientId), DateFilterFactory.createEmptyFilter());
		String appointmentId = appointments.get(0).getUniqueId();
		Appointment appointment = dao.fetchAppointment(createMockPatientIdentifier(patientId), appointmentId);
		assertTrue(NullChecker.isNotNullish(appointment.getClinicalNotes()));
	}

	@Test
	public void retrieveClinicInOrder() {
		AppointmentDataLayer dao = getDataLayer();
		String patientId = "D123401";
		Appointments appointments = dao.fetchAppointmentsByPatient(createMockPatientIdentifier(patientId), DateFilterFactory.createFilterFromDate("1/1/2000", "1/20/2012"));
		assertNotNull(appointments);

		Date previousDate = new Date();
		for (Appointment appointment : appointments) {
			assertTrue(appointment.getAppointmentStartDate().before(previousDate));
		}
	}
}
