package com.agilex.healthcare.mobilehealthplatform.datalayer.appointment;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.util.Calendar;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.junit.Ignore;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequest;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequestDetailCode;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequests;
import com.agilex.healthcare.mobilehealthplatform.domain.DetailCode;
import com.agilex.healthcare.mobilehealthplatform.domain.Facility;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentRequestProviderOption;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentRequestPurposeOfVisit;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentRequestStatus;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentRequestType;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentRequestVisitType;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullChecker;

@Ignore
public class AppointmentRequestDataLayerExcelTest {
	private String appointmentType;
	private String visitType;
	private String email;
	private String phoneNumber;
	private String status;
	private String optionDate1;
	private String optionTime1;
	private String optionDate2;
	private String optionTime2;
	private String optionDate3;
	private String optionTime3;
	private String purposeOfVisit;
	private String otherPurposeOfVisit;
	private String providerId;
	private String providerOption;
	private Date createdDate;
	private String firstName;
	private String lastName;
	private boolean secondRequest;
	private boolean textMessagingAllowed;
	private String textMessagingPhoneNumber;
	
	private Set<String> bestTimetoCall;
	private AppointmentRequests appointmentRequests;
	private AppointmentRequestDataLayer dao;
	
	private String patientId = "D123401";
	
	public AppointmentRequestDataLayerExcelTest() {
		dao = getDataLayer();
		init();
		appointmentRequests = dao.fetchAppointmentRequestsByPatient(patientId, DateFilterFactory.createEmptyFilter());
	}

	private void init() {
		appointmentType = AppointmentRequestType.MENTAL_HEALTH.getName();
		visitType = AppointmentRequestVisitType.OFFICE_VISIT.getName();
		email = "test@agilex.com";
		phoneNumber = "555-555-5555";
		Calendar calendar = Calendar.getInstance();
		createdDate = calendar.getTime();
		calendar.add(Calendar.DATE, 3);
		optionDate1 = DateHelper.formatDate(calendar.getTime());
		optionTime1 = "AM";
		calendar.add(Calendar.DATE, 1);
		optionDate2 = DateHelper.formatDate(calendar.getTime());
		optionTime2 = "PM";
		calendar.add(Calendar.DATE, 2);
		optionDate3 = DateHelper.formatDate(calendar.getTime());
		optionTime3 = "AM";
		status = AppointmentRequestStatus.SUBMITTED.getName();
		bestTimetoCall = new HashSet<String>();
		bestTimetoCall.add("9 AM - 11 AM");
		bestTimetoCall.add("11 AM - 1 PM");
		purposeOfVisit = AppointmentRequestPurposeOfVisit.OTHER.getName();
		otherPurposeOfVisit = "Other purpose of visit";
		providerId = "PROV1";
		providerOption = AppointmentRequestProviderOption.WILLING_TO_SEE_OTHER_PROVIDER.getName();
		firstName = "Gallow";
		lastName = "Younger";
		textMessagingAllowed = true;
		textMessagingPhoneNumber = "555-555-5555";
	}

	@Test
	public void retrievePatientAppointments() throws Exception {
		assertTrue(appointmentRequests.size()>0);
	}

	
	@Test
	public void retrieveAppointmentRequestsByApplyingFilter() throws Exception {
		AppointmentRequests appointmentRequests = dao.fetchAppointmentRequests(null);
		assertTrue(appointmentRequests.size()>0);
	}

	@Test
	public void retrieveAppointmentRequestsByApplyingFilterWithFacility1() throws Exception {
		AppointmentRequestFilter filter = new AppointmentRequestFilter();
		filter.setFacilityName("Jesse Brown VAMC");
		AppointmentRequests appointmentRequests = dao.fetchAppointmentRequests(filter);
		assertTrue(appointmentRequests.size()>0);
	}

	@Test
	public void retrieveAppointmentRequestsByApplyingFilterWithFacility1AndCaseInsensitive() throws Exception {
		AppointmentRequestFilter filter = new AppointmentRequestFilter();
		filter.setFacilityName("jesse brown vamc");
		AppointmentRequests appointmentRequests = dao.fetchAppointmentRequests(filter);
		assertTrue(appointmentRequests.size()>0);
	}

	@Test
	public void retrieveAppointmentRequestsByApplyingFilterWithFacility3() throws Exception {
		AppointmentRequestFilter filter = new AppointmentRequestFilter();
		filter.setParentSiteCode("537");
		filter.setStartDate(DateHelper.parseDate("5/1/2012"));
		filter.setEndDate(DateHelper.parseDate("5/25/2012"));
		AppointmentRequests appointmentRequests = dao.fetchAppointmentRequests(filter);
		assertEquals(1, appointmentRequests.size());
	}
	
	@Test
	public void ensureBookedAppointmentRequestsHaveAppointmentDateAndTimeAndDetailCode() throws Exception {
		for (AppointmentRequest request : appointmentRequests) {
			if (AppointmentRequestStatus.BOOKED.getName().equals(request.getStatus())) {
				assertBookedAppointmentRequest(request);
			}
		}
	}

	@Test
	public void ensureNotBookedAppointmentRequestsHaveDetailCode() throws Exception {
		for (AppointmentRequest request : appointmentRequests) {
			if (AppointmentRequestStatus.NOT_BOOKED.getName().equals(request.getStatus())) {
				assertNotBookedAppointmentRequest(request);
			}
		}
	}

	@Test
	public void ensureCancelledAppointmentRequestsHaveDetailCode() throws Exception {
		for (AppointmentRequest request : appointmentRequests) {
			if (AppointmentRequestStatus.CANCELLED.getName().equals(request.getStatus()) && NullChecker.isNotNullish(request.getAppointmentDate())) {
				assertCancelledAppointmentRequest(request);
			}
		}
	}

	@Test
	public void ensureSubmittedAppointmentRequestsDoesNotHaveDetailCode() throws Exception {
		for (AppointmentRequest request : appointmentRequests) {
			if (AppointmentRequestStatus.SUBMITTED.getName().equals(request.getStatus())) {
				assertTrue(NullChecker.isNullish(request.getAppointmentRequestDetailCode()));
			}
		}
	}
	
	@Test
	public void ensureAllRequestProviderOptionsNotNull() {
		for (AppointmentRequest request : appointmentRequests) {
			assertNotNull(request.getProviderOption());
		}
	}	
	
	@Test
	public void ensureRequestsHaveValidProviderOptions() {
		AppointmentRequestProviderOption[] validOptions = AppointmentRequestProviderOption.values();
		for (AppointmentRequest request : appointmentRequests) {
			assertNotNull(request.getProviderOption());
			if (!request.getProviderOption().equals("")) {
				boolean optionIsValid = false;
				for (AppointmentRequestProviderOption currentValidOption : validOptions) {
					if (request.getProviderOption().equals(currentValidOption.getName())) {
						optionIsValid = true;
						continue;
					}
				}
				assertTrue(optionIsValid);
			}
		}
	}
	
	private void assertCancelledAppointmentRequest(AppointmentRequest request) {
		List<AppointmentRequestDetailCode> arDetailCodes = request.getAppointmentRequestDetailCode();
		
		if(NullChecker.isNotNullish(request.getAppointmentDate())){
			if(arDetailCodes.size() == 2){
				 Iterator<AppointmentRequestDetailCode> iterator = arDetailCodes.iterator();
				 AppointmentRequestDetailCode arDetailCode1 = iterator.next();
				 AppointmentRequestDetailCode arDetailCode2 = iterator.next();
				
				 if(arDetailCode1.getCreatedDate().after(arDetailCode2.getCreatedDate())){
					 assertDetailCode(arDetailCode1.getDetailCode(), "DETCODE7");
					 assertDetailCode(arDetailCode2.getDetailCode(), "DETCODE8");
				 }else{
					 assertDetailCode(arDetailCode1.getDetailCode(), "DETCODE8");
					 assertDetailCode(arDetailCode2.getDetailCode(), "DETCODE7");
				 }
			}else if (arDetailCodes.size() == 1){
				AppointmentRequestDetailCode arDetailCode = arDetailCodes.iterator().next();
				assertTrue(arDetailCode.getDetailCode().getCode().equals("DETCODE8") || 
						arDetailCode.getDetailCode().getCode().equals("DETCODE6") || 
						arDetailCode.getDetailCode().getCode().equals("DETCODE5"));
			}else{
				assertTrue(false);
			}
		}else{
			// Cancelled by veteran before being booked by clerk
			assertTrue(NullChecker.isNullish(arDetailCodes));
		}
		
	}


	private void assertDetailCode(DetailCode detailCode, String code) {
		assertTrue(NullChecker.isNotNullish(detailCode.getProviderMessage()));
		assertTrue(NullChecker.isNotNullish(detailCode.getVeteranMessage()));
		assertEquals(code, detailCode.getCode());
	}

	private void assertNotBookedAppointmentRequest(AppointmentRequest request) {
		List<AppointmentRequestDetailCode> arDetailCodes = request.getAppointmentRequestDetailCode();
		assertEquals(1, arDetailCodes.size());
		DetailCode detailCode = arDetailCodes.iterator().next().getDetailCode();
		assertTrue(NullChecker.isNotNullish(detailCode.getProviderMessage()));
		assertTrue(NullChecker.isNotNullish(detailCode.getVeteranMessage()));

		if(request.isSecondRequest()){
			assertEquals("DETCODE4", detailCode.getCode());
		}else{
			assertEquals("DETCODE3", detailCode.getCode());
		}
	}

	private void assertBookedAppointmentRequest(AppointmentRequest request) {
		List<AppointmentRequestDetailCode> arDetailCodes = request.getAppointmentRequestDetailCode();
		assertEquals(1, arDetailCodes.size());
		DetailCode detailCode = arDetailCodes.iterator().next().getDetailCode();
		
		assertTrue(NullChecker.isNotNullish(detailCode.getProviderMessage()));
		assertTrue(NullChecker.isNotNullish(detailCode.getVeteranMessage()));
		
		if (NullChecker.isNullish(request.getAppointmentDate())){
			assertEquals("DETCODE2", detailCode.getCode());
		}else{
			assertTrue(NullChecker.isNotNullish(request.getAppointmentDate()));
			assertTrue(NullChecker.isNotNullish(request.getAppointmentTime()));
			assertEquals("DETCODE1", detailCode.getCode());
		}
	}

	@Test
	public void retrievePatientAppointmentNoResults() throws Exception {
		String patientId = "invalid";
		AppointmentRequests appointmentRequests = dao.fetchAppointmentRequestsByPatient(patientId, DateFilterFactory.createEmptyFilter());
		assertEquals(0, appointmentRequests.size());
	}

	@Test
	public void retrievePatientAppointment() throws Exception {
		String patientId = "D123401";
		AppointmentRequest appointmentRequest = dao.fetchAppointmentRequest(patientId, "1");
		assertNotNull(appointmentRequest);
	}
	
	@Test
	public void savePatientAppointment() throws Exception {
		String patientId = "D123401";
		AppointmentRequest originalRequest = createBasicAppointmentRequest(patientId);
		AppointmentRequest savedRequest = dao.saveAppointmentRequest(originalRequest);
		assertNotNull(savedRequest);
		assertNotNull(savedRequest.getDataIdentifier());
		assertNotNull(savedRequest.getDataIdentifier().getUniqueId());
		assertTrue(logicallyEqual(originalRequest, savedRequest));
	}
	
	@Test
	public void validateSaveSecondAppointmentRequest(){
		AppointmentRequest originalRequest = createBasicAppointmentRequest(patientId);
		originalRequest = dao.saveAppointmentRequest(originalRequest);

		AppointmentRequest secondRequest = createBasicAppointmentRequest(patientId);
		originalRequest.setSecondRequestSubmitted(true);
		secondRequest.setSecondRequest(true);
		secondRequest.setParentRequest(originalRequest);

		secondRequest = dao.saveAppointmentRequest(secondRequest);
		
		assertEquals(true, secondRequest.isSecondRequest());
		assertNull(secondRequest.getParentRequest());

		originalRequest = dao.fetchAppointmentRequest(patientId, originalRequest.getUniqueId());
		assertEquals(true, originalRequest.isSecondRequestSubmitted());
	}
	
	
	private AppointmentRequest createBasicAppointmentRequest(String userId) {

		String facilityAddress = "test address";
		String facilityCity = "test city";
		String facilityState = "test state";
		String facilityCode = "537";
		String facilityName = "test facility name";
		String facilityParentSiteCode = "test parent site code";
		String facilityType = "test facility type";
		Facility facility = createFacility(facilityAddress, facilityCity, facilityCode, facilityName, facilityParentSiteCode, facilityState, facilityType);
		
		AppointmentRequest appointmentRequest = new AppointmentRequest();
		appointmentRequest.setPatientId(userId);
		
		appointmentRequest.setAppointmentType(appointmentType);
		appointmentRequest.setVisitType(visitType);
		appointmentRequest.setFacility(facility);
		appointmentRequest.setEmail(email);
		appointmentRequest.setPhoneNumber(phoneNumber);
		appointmentRequest.setCreatedDate(createdDate);
		appointmentRequest.setOptionDate1(optionDate1);
		appointmentRequest.setOptionTime1(optionTime1);
		appointmentRequest.setOptionDate2(optionDate2);
		appointmentRequest.setOptionTime2(optionTime2);
		appointmentRequest.setOptionDate3(optionDate3);
		appointmentRequest.setOptionTime3(optionTime3);
		appointmentRequest.setStatus(status);
		appointmentRequest.setBestTimetoCall(bestTimetoCall);
		appointmentRequest.setPurposeOfVisit(purposeOfVisit);
		appointmentRequest.setOtherPurposeOfVisit(otherPurposeOfVisit);
		appointmentRequest.setProviderId(providerId);
		appointmentRequest.setProviderOption(providerOption);
		appointmentRequest.setTextMessagingAllowed(textMessagingAllowed);
		appointmentRequest.setTextMessagingPhoneNumber(textMessagingPhoneNumber);
		
		appointmentRequest.setPatient(new Patient());
		appointmentRequest.getPatient().setId(userId);
		appointmentRequest.getPatient().setFirstName(firstName);
		appointmentRequest.getPatient().setLastName(lastName);
		appointmentRequest.setSecondRequest(secondRequest);
		
		return appointmentRequest;
	}
	
	private Facility createFacility(String facilityAddress, String facilityCity, String facilityCode, String facilityName, String facilityParentSiteCode, String facilityState, String facilityType) {
		Facility facility = new Facility();
		facility.setAddress(facilityAddress);
		facility.setCity(facilityCity);
		facility.setFacilityCode(facilityCode);
		facility.setName(facilityName);
		facility.setParentSiteCode(facilityParentSiteCode);
		facility.setState(facilityState);
		facility.setType(facilityType);
		
		return facility;
	}
	
	private boolean logicallyEqual(AppointmentRequest original, AppointmentRequest other) {
		return (original.getPatientId().equalsIgnoreCase(other.getPatientId()) &&
				original.getAppointmentType().equalsIgnoreCase(other.getAppointmentType()) &&
				original.getVisitType().equalsIgnoreCase(other.getVisitType()) &&
				original.getFacility().getFacilityCode().equalsIgnoreCase(other.getFacility().getFacilityCode()) &&
				original.getEmail().equalsIgnoreCase(other.getEmail()) &&
				original.getPhoneNumber().equalsIgnoreCase(other.getPhoneNumber()) &&
				original.getCreatedDate().equals(other.getCreatedDate()) &&
				original.getOptionDate1().equalsIgnoreCase(other.getOptionDate1()) &&
				original.getOptionTime1().equalsIgnoreCase(other.getOptionTime1()) &&
				original.getOptionDate2().equalsIgnoreCase(other.getOptionDate2()) &&
				original.getOptionTime2().equalsIgnoreCase(other.getOptionTime2()) &&
				original.getOptionDate3().equalsIgnoreCase(other.getOptionDate3()) &&
				original.getOptionTime3().equalsIgnoreCase(other.getOptionTime3()) &&
				original.getStatus().equalsIgnoreCase(other.getStatus()) &&
				other.getBestTimetoCall().size() > 0 &&
				original.getBestTimetoCall().toString().equals(other.getBestTimetoCall().toString())  &&
				original.getProviderId().equals(other.getProviderId())  &&
				original.getProviderOption().equals(other.getProviderOption()) &&
				original.getOtherPurposeOfVisit().equalsIgnoreCase(other.getOtherPurposeOfVisit())  &&
				original.getPatient().getFirstName().equalsIgnoreCase(other.getPatient().getFirstName())  &&
				original.getPatient().getLastName().equalsIgnoreCase(other.getPatient().getLastName())  &&
				original.isSecondRequest() == other.isSecondRequest()  &&
				original.isTextMessagingAllowed() == other.isTextMessagingAllowed()  &&
				original.getTextMessagingPhoneNumber().equals(other.getTextMessagingPhoneNumber())  &&
				original.getPurposeOfVisit().equalsIgnoreCase(other.getPurposeOfVisit()));
	}

	private AppointmentRequestDataLayer getDataLayer() {
		AppointmentRequestDataLayer dao = new AppointmentRequestDataLayerExcel();
		return dao;
	}
}
