package com.agilex.healthcare.mobilehealthplatform.datalayer.xls;

import java.io.IOException;
import java.io.InputStream;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import jxl.Sheet;
import jxl.Workbook;
import jxl.WorkbookSettings;

public class ExcelRetriever {
	private static final org.apache.commons.logging.Log LOGGER = org.apache.commons.logging.LogFactory.getLog(ExcelRetriever.class);
	private DataElements data;
	Map<String, DomainDataElements> sheets = new ConcurrentHashMap<String, DomainDataElements>(); // SheetName -> DataElements

	public ExcelRetriever(String patientId, String domain) {
		loadDataForPatient(patientId, domain);
	}
	
	public ExcelRetriever(String patientId) {
		loadSheetsForPatient(patientId);
	}
	
	public Map<String, DomainDataElements> getSheetCache() {
		return this.sheets;
	}

	public DataElements getData() {
		return this.data;
	}

	private Map<String, DomainDataElements> loadSheetsForPatient(String patientId) {
		LOGGER.debug(String.format("Loading all data sheets for patient %s", patientId));
		Workbook workbook = getWorkbook(patientId);
		
		DomainDataElements domainDataElements = new DomainDataElements();
		if (workbook != null) {
			int numberOfSheets = workbook.getNumberOfSheets();
			for (int sheetNumber = 0; sheetNumber < numberOfSheets; sheetNumber++) {
				Sheet sheet = workbook.getSheet(sheetNumber);
				String sheetName = sheet.getName();
				
				DataElements sheetDataElements = readDataFromSheet(sheet);
				domainDataElements.put(sheetName, sheetDataElements);
				sheets.put(patientId, domainDataElements);
			}

			workbook.close();
		}
		return sheets;
	}

	/**
	 * Please use {@link #loadSheetsForPatient(String)}, which caches all sheets while the file is open.
	 * 
	 */
	@Deprecated
	private void loadDataForPatient(String patientId, String domain) {
		LOGGER.debug(String.format("beging load for patient %s, domain %s", patientId, domain));
		Workbook workbook = getWorkbook(patientId);
		if (workbook != null) {
			int numberOfSheets = workbook.getNumberOfSheets();

			Sheet sheet = null;

			for (int sheetNumber = 0; sheetNumber < numberOfSheets; sheetNumber++) {
				Sheet possibleSheet = workbook.getSheet(sheetNumber);
				if (possibleSheet.getName().contentEquals(domain)) {
					sheet = possibleSheet;
				}
			}

			if (sheet != null) {
				LOGGER.debug(String.format("found workbook sheet, load for patient %s, domain %s", patientId, domain));
				data = readDataFromSheet(sheet);
			}

			LOGGER.debug("close workbook");
			workbook.close();
		}

	}

	private Workbook getWorkbook(String patientId) {
		Workbook workbook = null;
		InputStream resourceStream = null;
		try {
			String resourceName = getResourceName(patientId);
			LOGGER.debug("Ready to open resource to read data.  Resource=" + resourceName);
			resourceStream = this.getClass().getResourceAsStream(resourceName);
			if (resourceStream != null) {
				LOGGER.debug("Ready to open resource by stream.  " + resourceStream);
				WorkbookSettings ws = new WorkbookSettings();
				ws.setGCDisabled(true);
				workbook = Workbook.getWorkbook(resourceStream, ws);
			} else {
				LOGGER.warn("unable to find resource: " + resourceStream);
			}

		} catch (Exception e) {
			// if no workbook found, then we will return no workbook and an
			// empty set of labs
			LOGGER.error("Error loading workbook for " + patientId, e);
		} finally {
			if (resourceStream != null) {
				try {
					resourceStream.close();
				} catch (IOException e) {
					LOGGER.error("Error closing workbook stream for " + patientId, e);
				}
			}
		}

		return workbook;

	}

	private String getResourceName(String patientId) {
		return String.format("%s.xls", patientId);
	}

	private DataElements readDataFromSheet(Sheet sheet) {
		DataElements retrievedData = new DataElements();
		int totalRows = sheet.getRows();
		
		int rowNumber = 1;
		while (rowNumber < totalRows) {
			DataElement r = readRow(sheet, rowNumber);
			retrievedData.add(r);
			rowNumber++;
		}
		return retrievedData;
	}

	private DataElement readRow(Sheet sheet, int rowNumber) {
		int totalColumns = sheet.getColumns();

		DataElement r = new DataElement();

		int columnNumber = 0;
		while (columnNumber < totalColumns) {
			String columnHeader = getColumnHeader(columnNumber, sheet);
			String value = getValue(columnNumber, rowNumber, sheet);
			LOGGER.debug(String.format("read row %s, column %s (#%s), value %s", rowNumber, columnHeader, columnNumber, value));
			r.put(columnHeader, value);
			columnNumber++;
		}

		return r;
	}

	private String getValue(int column, int row, Sheet sheet) {
		return sheet.getCell(column, row).getContents();
	}

	private String getColumnHeader(int column, Sheet sheet) {
		final int headerRow = 0;
		String header = getValue(column, headerRow, sheet);
		if (header != null)
			header = header.toLowerCase();
		return header;
	}
}
