package com.agilex.healthcare.mobilehealthplatform.datalayer.vitals;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

import com.agilex.healthcare.mobilehealthplatform.datalayer.vital.VitalDataLayer;
import com.agilex.healthcare.mobilehealthplatform.datalayer.vital.VitalEntryFilter;
import com.agilex.healthcare.mobilehealthplatform.datalayer.vital.VitalDataLayerFilterer;
import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.domain.code.ObservationTypeCode;
import com.agilex.healthcare.mobilehealthplatform.domain.code.VitalSectionCode;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.NullSafeStringComparer;


// NOTE: This should be used for Provider vitals while excel should be used for Patient-entered (SED) Vitals. 
// The format of blood pressure is slightly different (pulse is an observation for SED Vitals but separated out for provider data sources)

public class VitalDataLayerMock implements VitalDataLayer {

	private static Map<String, VitalEntries> mockVitalDataStorage;
	static Integer mockGeneratedEntryId = 0;

	static {
		initializeStorage();
	}

	private static synchronized void initializeStorage() {
		mockVitalDataStorage = new ConcurrentHashMap<String, VitalEntries>();
        
		String[] patientIds = { "D123401", "PATID65", "PATID66", "PATID67", "PATID68", "PATID69", "PATID70", "PATID71", "PATID72", "PATID73", "PATID74", "PATID76", "0010010010", "0010010011" };

        for(String patientId: patientIds) {
            
            VitalEntry vitalEntry = new VitalEntry();
            vitalEntry.setSection(VitalSectionCode.Pulse);
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 5));
            vitalEntry.setQualifier("Standing");
            vitalEntry.setPatientId(patientId);
            VitalObservation observation = new VitalObservation();
            observation.setType(ObservationTypeCode.HEART_BEAT);
            observation.setValue("60");
            observation.setValueUnits("BPM");
            vitalEntry.addObservation(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setSection(VitalSectionCode.BloodPressure);
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setQualifier("Standing");
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 5));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.INTRAVASCULAR_SYSTOLIC);
            observation.setValue("130");
            observation.setValueUnits("mm(hg)");
            vitalEntry.getVitalObservations().add(observation);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.INTRAVASCULAR_DIASTOLIC);
            observation.setValue("80");
            observation.setValueUnits("mm(hg)");
            vitalEntry.getVitalObservations().add(observation);
            vitalEntry.setNotes("test note");
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.Pulse);
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 4));
            vitalEntry.setQualifier("Lying");
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.HEART_BEAT);
            observation.setValue("55");
            observation.setValueUnits("BPM");
            vitalEntry.addObservation(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.BloodPressure);
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 4));
            vitalEntry.setQualifier("Lying");
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.INTRAVASCULAR_SYSTOLIC); 
            observation.setValue("125");
            observation.setValueUnits("mm(hg)");
            vitalEntry.getVitalObservations().add(observation);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.INTRAVASCULAR_DIASTOLIC);
            observation.setValue("75");
            observation.setValueUnits("mm(hg)");
            vitalEntry.getVitalObservations().add(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.Height);
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 5));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.HEIGHT);
            observation.setValue("60");
            observation.setValueUnits("inches");
            vitalEntry.getVitalObservations().add(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.Height);
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 4));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.HEIGHT);
            observation.setValue("60");
            observation.setValueUnits("inches");
            vitalEntry.getVitalObservations().add(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.Temperature);
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 4));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.BODY_TEMPERATURE);
            observation.setValue("98.6");
            observation.setValueUnits("degrees");
            vitalEntry.getVitalObservations().add(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.Mood);
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 4));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.MOOD);
            observation.setValue("happy");
            vitalEntry.getVitalObservations().add(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.Pain);
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 4));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.PAIN);
            observation.setValue("3");
            vitalEntry.getVitalObservations().add(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.Pain);
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 4));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.PAIN);
            observation.setValue("7");
            vitalEntry.getVitalObservations().add(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.Temperature);
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 4));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.BODY_TEMPERATURE);
            observation.setValue("101");
            observation.setValueUnits("degrees");
            vitalEntry.getVitalObservations().add(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.Temperature);
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 5));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.BODY_TEMPERATURE);
            observation.setValue("101");
            observation.setValueUnits("degrees");
            vitalEntry.getVitalObservations().add(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.Temperature);
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 4));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.BODY_TEMPERATURE);
            observation.setValue("101");
            observation.setValueUnits("degrees");
            vitalEntry.getVitalObservations().add(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.Temperature);
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 3));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.BODY_TEMPERATURE);
            observation.setValue("101");
            observation.setValueUnits("degrees");
            vitalEntry.getVitalObservations().add(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.Weight);
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 5));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.WEIGHT);
            observation.setValue("170");
            observation.setValueUnits("lb");
            vitalEntry.getVitalObservations().add(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.Weight);
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 4));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.WEIGHT);
            observation.setValue("165");
            observation.setValueUnits("lb");
            vitalEntry.getVitalObservations().add(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.Weight);
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 3));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.WEIGHT);
            observation.setValue("160");
            observation.setValueUnits("lb");
            vitalEntry.getVitalObservations().add(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.Respiration);
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 5));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.RESPIRATION);
            observation.setValue("55");
            vitalEntry.getVitalObservations().add(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.Respiration);
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 4));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.RESPIRATION);
            observation.setValue("55");
            vitalEntry.getVitalObservations().add(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.Respiration);
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 3));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.RESPIRATION);
            observation.setValue("54");
            vitalEntry.getVitalObservations().add(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.Pulse);
            vitalEntry.setQualifier("Standing");
            vitalEntry.setNotes("Test note");
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 3));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.HEART_BEAT);
            observation.setValue("70");
            observation.setValueUnits("BPM");
            vitalEntry.addObservation(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.BloodPressure);
            vitalEntry.setPatientId(patientId);
            vitalEntry.setQualifier("Standing");
            vitalEntry.setNotes("Test note");
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 3));
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.INTRAVASCULAR_SYSTOLIC);
            observation.setValue("120");
            observation.setValueUnits("mm(hg)");
            vitalEntry.addObservation(observation);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.INTRAVASCULAR_DIASTOLIC);
            observation.setValue("85");
            observation.setValueUnits("mm(hg)");
            vitalEntry.addObservation(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.Pulse);
            vitalEntry.setPatientId(patientId);
            vitalEntry.setQualifier("Standing");
            vitalEntry.setNotes("Test note");
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 4));
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.HEART_BEAT);
            observation.setValue("70");
            observation.setValueUnits("BPM");
            vitalEntry.addObservation(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.BloodPressure);
            vitalEntry.setQualifier("Standing");
            vitalEntry.setNotes("Test note");
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 4));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.INTRAVASCULAR_SYSTOLIC);
            observation.setValue("120");
            observation.setValueUnits("mm(hg)");
            vitalEntry.getVitalObservations().add(observation);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.INTRAVASCULAR_DIASTOLIC);
            observation.setValue("85");
            observation.setValueUnits("mm(hg)");
            vitalEntry.getVitalObservations().add(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.Pulse);
            vitalEntry.setQualifier("Standing");
            vitalEntry.setNotes("Test note");
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 5));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.HEART_BEAT);
            observation.setValue("70");
            observation.setValueUnits("BPM");
            vitalEntry.addObservation(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.BloodPressure);
            vitalEntry.setQualifier("Standing");
            vitalEntry.setNotes("Test note");
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 5));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.INTRAVASCULAR_SYSTOLIC);
            observation.setValue("120");
            observation.setValueUnits("mm(hg)");
            vitalEntry.addObservation(observation);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.INTRAVASCULAR_DIASTOLIC);
            observation.setValue("85");
            observation.setValueUnits("mm(hg)");
            vitalEntry.addObservation(observation);
            setVitalEntry(vitalEntry);

            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.PulseOximetry);
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 4));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.PULSE_OXIMETRY);
            observation.setValue("95");
            observation.setValueUnits("%");
            vitalEntry.addObservation(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.PulseOximetry);
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 5));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.PULSE_OXIMETRY);
            observation.setValue("98");
            observation.setValueUnits("%");
            vitalEntry.addObservation(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.PulseOximetry);
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 3));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.PULSE_OXIMETRY);
            observation.setValue("99");
            observation.setValueUnits("%");
            vitalEntry.addObservation(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.PulseOximetry);
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 5));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.PULSE_OXIMETRY);
            observation.setValue("96");
            observation.setValueUnits("%");
            vitalEntry.addObservation(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.PulseOximetry);
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 5));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.PULSE_OXIMETRY);
            observation.setValue("97");
            observation.setValueUnits("%");
            vitalEntry.addObservation(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.PulseOximetry);
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 3));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.PULSE_OXIMETRY);
            observation.setValue("98");
            observation.setValueUnits("%");
            vitalEntry.addObservation(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.PulseOximetry);
            vitalEntry.setEntryDate(DateHelper.minusDays(new Date(), 5));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.PULSE_OXIMETRY);
            observation.setValue("99");
            observation.setValueUnits("%");
            vitalEntry.addObservation(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.PulseOximetry);
            vitalEntry.setEntryDate(DateHelper.parseDate("10/10/2011"));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.PULSE_OXIMETRY);
            observation.setValue("96");
            observation.setValueUnits("%");
            vitalEntry.addObservation(observation);
            setVitalEntry(vitalEntry);
            
            vitalEntry = new VitalEntry();
            vitalEntry.setDataIdentifier("mock", UUID.randomUUID().toString());
            vitalEntry.setSection(VitalSectionCode.PulseOximetry);
            vitalEntry.setEntryDate(DateHelper.parseDate("10/10/2010"));
            vitalEntry.setPatientId(patientId);
            observation = new VitalObservation();
            observation.setType(ObservationTypeCode.PULSE_OXIMETRY);
            observation.setValue("97");
            observation.setValueUnits("%");
            vitalEntry.addObservation(observation);
            setVitalEntry(vitalEntry);
        }
	}

	private static synchronized void setVitalEntry(VitalEntry vitalEntry) {
		if (NullChecker.isNullish(vitalEntry.getId())) {
			generateAndPopulateIdentifier(vitalEntry);
		}

		VitalEntries patientVitals = mockVitalDataStorage.get(vitalEntry.getPatientId());

		if (patientVitals == null) {
			patientVitals = new VitalEntries();
			mockVitalDataStorage.put(vitalEntry.getPatientId(), patientVitals);
		}

		VitalEntry existingVital = null;
		for (VitalEntry potentialMatch : patientVitals) {
			if (NullSafeStringComparer.areEqual(potentialMatch.getId(), vitalEntry.getId())) {
				existingVital = potentialMatch;
				break;
			}
		}

		if (existingVital != null) {
			patientVitals.remove(existingVital);
		}

		patientVitals.add(vitalEntry);
	}

	public VitalEntries getVitalEntries(String patientId) {
		return getVitalEntries(patientId, null);
	}

	@Override
	public VitalEntries getVitalEntries(String patientId, VitalEntryFilter filter) {
		if (filter == null) {
			filter = new VitalEntryFilter();
		}

		VitalEntries vitalEntriesForPatient = mockVitalDataStorage.get(patientId);
		if (vitalEntriesForPatient == null) {
			vitalEntriesForPatient = new VitalEntries();
		}

		VitalEntries filteredVitalEntries = VitalDataLayerFilterer.applyFilter(vitalEntriesForPatient, filter);
		return filteredVitalEntries;
	}

	@Override
	public VitalEntry saveVitalEntry(VitalEntry vitalEntry) {
		setVitalEntry(vitalEntry);
		return vitalEntry;
	}

	private static void generateAndPopulateIdentifier(VitalEntry vitalEntry) {
		vitalEntry.setDataIdentifier("mock", (++mockGeneratedEntryId).toString());
	}

	@Override
	public VitalEntry getVitalEntry(String patientId, String vitalId) {
		VitalEntry match = null;
		VitalEntries vitalEntriesForPatient = getVitalEntries(patientId);
		for (VitalEntry potentialMatch : vitalEntriesForPatient) {
			if (NullSafeStringComparer.areEqual(vitalId, potentialMatch.getId())) {
				match = potentialMatch;
				break;
			}
		}
		return match;
	}

	@Override
	public void deleteVitalEntry(VitalEntry vitalEntry) {
		if (vitalEntry == null)
			throw new IllegalArgumentException("Vital entry unexpectedly null, cannot delete");
		if (NullChecker.isNullish(vitalEntry.getPatientId()))
			throw new IllegalArgumentException("Vital entry does not contain a patient identifier, cannot delete");
		if (NullChecker.isNullish(vitalEntry.getId()))
			throw new IllegalArgumentException("Vital entry does not contain an identifier, cannot delete");

		VitalEntry existingVital = null;
		VitalEntries patientVitals = mockVitalDataStorage.get(vitalEntry.getPatientId());
		for (VitalEntry potentialMatch : patientVitals) {
			if (NullSafeStringComparer.areEqual(potentialMatch.getId(), vitalEntry.getId())) {
				existingVital = potentialMatch;
				break;
			}
		}

		if (existingVital != null) {
			patientVitals.remove(existingVital);
		}
	}

	@Override
	public VitalEntries fetchVitalEntries(PatientIdentifier patientIdentifier, VitalEntryFilter filter) {
		return getVitalEntries(patientIdentifier.getUniqueId(), filter);
	}

	@Override
	public VitalEntry fetchVitalEntry(PatientIdentifier patientIdentifier, String vitalId) {
		return getVitalEntry(patientIdentifier.getUniqueId(), vitalId);
	}

}
