package com.agilex.healthcare.mobilehealthplatform.datalayer.vitals;

import org.apache.commons.lang.NotImplementedException;

import com.agilex.healthcare.mobilehealthplatform.MockDataStorage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.vital.*;
import com.agilex.healthcare.mobilehealthplatform.datalayer.xls.*;
import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.utility.NullSafeStringComparer;

//NOTE: This should be used for Provider vitals while excel should be used for Patient-entered (SED) Vitals. 
//The format of blood pressure is slightly different (pulse is an observation for SED Vitals but separated out for provider data sources)
public class VitalDataLayerExcel implements VitalDataLayer {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(VitalDataLayerExcel.class);

	private static MockDataStorage<VitalEntries, VitalEntry> data = new MockDataStorage<VitalEntries, VitalEntry>() {
		@Override
		protected VitalEntries createDataCollection() {
			return new VitalEntries();
		}
	};

	private static final String COLUMN_PATIENTID = "PatientId";
	private static final String COLUMN_VITALID = "VitalId";
	private static final String COLUMN_SECTION = "Section";
	private static final String COLUMN_ENTRYDATE = "EntryDate";
	private static final String COLUMN_QUALIFIER = "Qualifier";
	private static final String COLUMN_NOTES = "Notes";
	private static final String COLUMN_OBSERVATIONTYPE = "ObservationType";
	private static final String COLUMN_OBSERVATIONVALUE = "ObservationValue";
	private static final String COLUMN_OBSERVATIONVALUEUNITS = "ObservationValueUnits";
	private static final String domain = "Vitals";

	public VitalEntries getVitalEntries(String patientId) {
		logger.debug("get vitals for patient " + patientId);
		if (!data.dataExists(patientId)) {
			DataElementDataStorage storage = DataElementDataStorage.getInstance();
			DataElements dataElements = storage.getData(patientId, domain);
			data.setDataElements(createVitalEntries(dataElements));
		}
		return data.getDataForPatient(patientId);
	}

	@Override
	public VitalEntries getVitalEntries(String patientId, VitalEntryFilter filter) {
		logger.debug("get vitals with filter for patient " + patientId);

		if (filter == null) {
			filter = new VitalEntryFilter();
		}
		logger.debug("about to get vitals with filter for patient " + patientId);

		VitalEntries vitalEntriesForPatient = getVitalEntries(patientId);
		
		if (vitalEntriesForPatient == null) {
			vitalEntriesForPatient = new VitalEntries();
		}
		logger.debug("vital entries found : " + vitalEntriesForPatient.size());

		VitalEntries filteredVitalEntries = VitalDataLayerFilterer.applyFilter(vitalEntriesForPatient, filter);
		
		return filteredVitalEntries;
	}

	private VitalEntries createVitalEntries(DataElements dataElements) {
		VitalEntries vitalEntries = new VitalEntries();

		for (DataElement dataElement : dataElements) {
			if (isDataElementVitalEntry(dataElement)) {
				VitalEntry vitalEntry = createVitalEntry(dataElement);
				vitalEntries.add(vitalEntry);
			} else {
				VitalObservation observation = createObservation(dataElement);
				VitalEntry vitalEntry = getVitalEntry(vitalEntries, observation.getVitalEntry().getId());
				vitalEntry.getVitalObservations().add(observation);
			}

		}

		return vitalEntries;
	}

	private VitalEntry getVitalEntry(VitalEntries vitalEntries, String id) {
		VitalEntry match = null;
		for (VitalEntry potentialMatch : vitalEntries) {
			if (potentialMatch.getId().contentEquals(id)) {
				match = potentialMatch;
				break;
			}
		}
		return match;
	}

	private VitalEntry createVitalEntry(DataElement dataElement) {
		VitalEntry vitalEntry = new VitalEntry();

		vitalEntry.setPatientId(dataElement.getValue(COLUMN_PATIENTID));
		vitalEntry.setId(dataElement.getValue(COLUMN_VITALID));
		vitalEntry.setSection(dataElement.getValue(COLUMN_SECTION));
		vitalEntry.setEntryDate(dataElement.getValueAsDateTime(COLUMN_ENTRYDATE));
		vitalEntry.setNotes(dataElement.getValue(COLUMN_NOTES));
		vitalEntry.setQualifier(dataElement.getValue(COLUMN_QUALIFIER));

		return vitalEntry;

	}

	private VitalObservation createObservation(DataElement dataElement) {
		VitalObservation vitalObservation = new VitalObservation();
		VitalEntry vitalEntry = new VitalEntry();
		vitalEntry.setId(dataElement.getValue(COLUMN_VITALID));
		vitalObservation.setVitalEntry(vitalEntry);
		vitalObservation.setType(dataElement.getValue(COLUMN_OBSERVATIONTYPE));
		vitalObservation.setValue(dataElement.getValue(COLUMN_OBSERVATIONVALUE));
		vitalObservation.setValueUnits(dataElement.getValue(COLUMN_OBSERVATIONVALUEUNITS));
		return vitalObservation;
	}

	private boolean isDataElementVitalEntry(DataElement dataElement) {
		return dataElement.getValue(COLUMN_SECTION).contentEquals("") == false;
	}

	@Override
	public VitalEntry saveVitalEntry(VitalEntry vitalEntry) {
		throw new NotImplementedException("This method is not available for data entered by providers");
	}

	@Override
	public VitalEntry getVitalEntry(String patientId, String vitalId) {
		VitalEntry match = null;
		VitalEntries vitalEntriesForPatient = getVitalEntries(patientId);
		for (VitalEntry potentialMatch : vitalEntriesForPatient) {
			if (NullSafeStringComparer.areEqual(vitalId, potentialMatch.getId())) {
				match = potentialMatch;
				break;
			}
		}
		return match;
	}

	@Override
	public void deleteVitalEntry(VitalEntry vitalEntry) {
		throw new NotImplementedException("This method is not available for data entered by providers");
	}
	

	@Override
	public VitalEntries fetchVitalEntries(PatientIdentifier patientIdentifier, VitalEntryFilter filter) {
		return getVitalEntries(patientIdentifier.getUniqueId(), filter);
	}

	@Override
	public VitalEntry fetchVitalEntry(PatientIdentifier patientIdentifier, String vitalId) {
		return getVitalEntry(patientIdentifier.getUniqueId(), vitalId);
	}

}
