package com.agilex.healthcare.mobilehealthplatform.datalayer.securemessage;

import java.util.Collections;
import java.util.Date;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.log4j.Logger;

import com.agilex.healthcare.mobilehealthplatform.domain.SecureMessage;
import com.agilex.healthcare.mobilehealthplatform.domain.SecureMessageCode;
import com.agilex.healthcare.mobilehealthplatform.domain.SecureMessageContact;
import com.agilex.healthcare.mobilehealthplatform.domain.SecureMessageContacts;
import com.agilex.healthcare.mobilehealthplatform.domain.SecureMessageFolder;
import com.agilex.healthcare.mobilehealthplatform.domain.SecureMessageFolders;
import com.agilex.healthcare.mobilehealthplatform.domain.SecureMessageUser;
import com.agilex.healthcare.mobilehealthplatform.domain.SecureMessages;
import com.agilex.healthcare.mobilehealthplatform.domain.code.AssigningAuthorityCode;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.NullSafeStringComparer;
import com.agilex.healthcare.utility.Paginator;

public class SecureMessageDataLayerMock implements SecureMessageDataLayer {

	Logger logger = Logger.getLogger(SecureMessageDataLayerMock.class);

	Map<String, SecureMessageUser> userStorage = new ConcurrentHashMap<String, SecureMessageUser>();
	Map<String, SecureMessageContacts> contactStorage = new ConcurrentHashMap<String, SecureMessageContacts>();
	Map<String, SecureMessageFolders> folderStorage = new ConcurrentHashMap<String, SecureMessageFolders>();
	Map<String, SecureMessages> messageStorage = new ConcurrentHashMap<String, SecureMessages>();
	
	AtomicInteger addresseeId = new AtomicInteger(0);
	AtomicInteger addresseeOplock = new AtomicInteger(0);

	public SecureMessageDataLayerMock() {
		initPatient("D123401", "D123401");
		initPatient("D123401", "71");
		initPatient("P123401", "72");
		initPatient("PDOC123", "73");
		initPatient("10000000219", "72");
		initPatient("PATID76", "72");
		initPatient("983", "72");
		initPatient("9025", "72");
		initPatient("PVMS1234", "72");
		initPatient("PVMS5678", "72");
	}

	private void initPatient(String userId, String secureMessageUserId) {
		initUser(userId, secureMessageUserId);
		initFolders(secureMessageUserId);
		initContacts(secureMessageUserId);
		initMessages(secureMessageUserId);
	}

	private void initUser(String userId, String secureMessageUserId) {
		SecureMessageUser user = new SecureMessageUser();
		user.setFirstName("zztest");
		user.setLastName("patient01");
		user.setMiddleName("a.");
		user.setPatientIdentifier(AssigningAuthorityCode.SECURE_MESSAGE, secureMessageUserId);
		user.setDataIdentifier("mock", UUID.randomUUID().toString());
		user.setEmail("Patient.User@email.gov");
		user.setUsername("zztest.patient01");

		userStorage.put(userId, user);
	}

	private void initFolders(String userId) {
		SecureMessageFolders folders = new SecureMessageFolders();

		SecureMessageFolder inbox = new SecureMessageFolder();
		inbox.setSystemFolder(true);
		inbox.setMovable(true);
		inbox.setDataIdentifier("mock", SecureMessageCode.FOLDER_ID_INBOX.toString());
		inbox.setPatientIdentifier(AssigningAuthorityCode.SECURE_MESSAGE, userId);
		inbox.setName("Inbox");
		folders.add(inbox);

		SecureMessageFolder sent = new SecureMessageFolder();
		sent.setSystemFolder(true);
		sent.setDataIdentifier("mock", SecureMessageCode.FOLDER_ID_SENT.toString());
		sent.setPatientIdentifier(AssigningAuthorityCode.SECURE_MESSAGE, userId);
		sent.setName("Sent");
		folders.add(sent);

		SecureMessageFolder drafts = new SecureMessageFolder();
		drafts.setSystemFolder(true);
		drafts.setDataIdentifier("mock", SecureMessageCode.FOLDER_ID_DRAFT.toString());
		drafts.setPatientIdentifier(AssigningAuthorityCode.SECURE_MESSAGE, userId);
		drafts.setName("Drafts");
		folders.add(drafts);

		SecureMessageFolder deleted = new SecureMessageFolder();
		deleted.setSystemFolder(true);
		deleted.setMovable(true);
		deleted.setDataIdentifier("mock", SecureMessageCode.FOLDER_ID_DELETED.toString());
		deleted.setPatientIdentifier(AssigningAuthorityCode.SECURE_MESSAGE, userId);
		deleted.setName("Deleted");
		folders.add(deleted);

		SecureMessageFolder completed = new SecureMessageFolder();
		completed.setSystemFolder(true);
		completed.setMovable(false);
		completed.setDataIdentifier("mock", SecureMessageCode.FOLDER_ID_COMPLETED.toString());
		completed.setPatientIdentifier(AssigningAuthorityCode.SECURE_MESSAGE, userId);
		completed.setName("Completed");
		folders.add(completed);

		SecureMessageFolder pharmacy = new SecureMessageFolder();
		pharmacy.setSystemFolder(false);
		pharmacy.setMovable(true);
		pharmacy.setDataIdentifier("mock", "3000");
		pharmacy.setPatientIdentifier(AssigningAuthorityCode.SECURE_MESSAGE, userId);
		pharmacy.setName("Pharmacy");
		folders.add(pharmacy);

		SecureMessageFolder archive = new SecureMessageFolder();
		archive.setSystemFolder(false);
		archive.setMovable(true);
		archive.setDataIdentifier("mock", "3001");
		archive.setPatientIdentifier(AssigningAuthorityCode.SECURE_MESSAGE, userId);
		archive.setName("Archive");
		folders.add(archive);

		folderStorage.put(userId, folders);
	}

	private void initContacts(String userId) {
		SecureMessageContacts contacts = new SecureMessageContacts();

		if ("72".equalsIgnoreCase(userId) || "73".equalsIgnoreCase(userId)) {
			SecureMessageContact mhv = new SecureMessageContact();
			mhv.setName("zztest patient01");
			mhv.setDataIdentifier("mock-aa", "D123401");
			mhv.setSourceSystem("SM");
			mhv.setPatientIdentifier(AssigningAuthorityCode.SECURE_MESSAGE, userId);
			contacts.add(mhv);

			SecureMessageContact advocate = new SecureMessageContact();
			advocate.setName("zztest patient26");
			advocate.setDataIdentifier("mock-aa", "PATID26");
			advocate.setSourceSystem("SM");
			advocate.setPatientIdentifier(AssigningAuthorityCode.SECURE_MESSAGE, userId);
			contacts.add(advocate);

			contactStorage.put(userId, contacts);

		} else {
			SecureMessageContact travel = new SecureMessageContact();
			travel.setName("Ask a Patient Travel Question - Test VAMC");
			travel.setDataIdentifier("mock", "979");
			travel.setSourceSystem("SM");
			travel.setPatientIdentifier(AssigningAuthorityCode.SECURE_MESSAGE, userId);
			contacts.add(travel);

			SecureMessageContact pharmacy = new SecureMessageContact();
			pharmacy.setName("Ask a Pharmacist - Test VAMC");
			pharmacy.setDataIdentifier("mock", "4001");
			pharmacy.setSourceSystem("SM");
			pharmacy.setPatientIdentifier(AssigningAuthorityCode.SECURE_MESSAGE, userId);
			contacts.add(pharmacy);

			SecureMessageContact chaplain = new SecureMessageContact();
			chaplain.setName("Chaplain Service - Test VAMC");
			chaplain.setDataIdentifier("mock", "4002");
			chaplain.setSourceSystem("SM");
			chaplain.setPatientIdentifier(AssigningAuthorityCode.SECURE_MESSAGE, userId);
			contacts.add(chaplain);

			SecureMessageContact mhv = new SecureMessageContact();
			mhv.setName("My HealtheVet - Test VAMC");
			mhv.setDataIdentifier("mock", "4003");
			mhv.setSourceSystem("SM");
			mhv.setPatientIdentifier(AssigningAuthorityCode.SECURE_MESSAGE, userId);
			contacts.add(mhv);

			SecureMessageContact advocate = new SecureMessageContact();
			advocate.setName("Patient Advocate - Test VAMC");
			advocate.setDataIdentifier("mock", "4004");
			advocate.setSourceSystem("SM");
			advocate.setPatientIdentifier(AssigningAuthorityCode.SECURE_MESSAGE, userId);
			contacts.add(advocate);

			contactStorage.put(userId, contacts);
		}
	}

	private void initMessages(String userId) {
		SecureMessages messages = new SecureMessages();
		
		int inboxMessages = 30;
		SecureMessage inbox = null;
		for (int i = 0; i < inboxMessages; i++) {
			inbox = new SecureMessage();
			inbox.setAddresseeId(Integer.toString(addresseeId.addAndGet(1)));
			inbox.setAddresseeOplock("0");
			inbox.setThreadId(UUID.randomUUID().toString());
			inbox.setThreadOplock(UUID.randomUUID().toString());
			inbox.setDateSent(DateHelper.parseDate("06/"+ Integer.toString(i) + "/2012"));
			inbox.setFolder("Inbox");
			inbox.setFolderId(SecureMessageCode.FOLDER_ID_INBOX.toString());
			inbox.setDataIdentifier("mock", "000"+Integer.toString(i));
			inbox.setMessageBody("Inbox Message "+Integer.toString(i));
			inbox.setMessageOplock("0");
			inbox.setPatientIdentifier(AssigningAuthorityCode.SECURE_MESSAGE, userId);
			inbox.setSubject("Inbox Subject "+Integer.toString(i)+ " extra text 30 characters");
			inbox.setSubjectCategory(String.valueOf(SecureMessageCode.SUBJECT_CATEGORY_ID_GENERAL));
			inbox.setFrom("Chaplain Service - Test VAMC");
			inbox.setFromId("4002");
			inbox.setTo("zztest patient01");
			inbox.setToId(userId);
			inbox.setMailGroup("4002");
			if ((i%2) == 0) {
				inbox.setReadDate(DateHelper.parseDate("06/"+ Integer.toString(i) + "/2012"));
			}
			messages.add(inbox);
		}
        
		int draftMessages = 30;
		SecureMessage draft = null;
//		SecureMessage draft = new SecureMessage();
		for (int i = 0; i < draftMessages; i++) {
			draft = new SecureMessage();
			draft.setAddresseeId(Integer.toString(addresseeId.addAndGet(1)));
			draft.setAddresseeOplock("0");
			draft.setThreadId(UUID.randomUUID().toString());
			draft.setThreadOplock(UUID.randomUUID().toString());
//			draft.setDateSent(DateHelper.parseDate("06/"+Integer.toString(i)+"/2012"));
			draft.setFolder("Drafts");
			draft.setFolderId(SecureMessageCode.FOLDER_ID_DRAFT.toString());
			draft.setDataIdentifier("mock", "20"+Integer.toString(i));
			draft.setMessageBody("Draft Message "+Integer.toString(i));
			draft.setSubject("Appointment for Next Month - Appt. Subject Category");
			draft.setSubjectCategory(String.valueOf(SecureMessageCode.SUBJECT_CATEGORY_ID_APPOINTMENT));
			draft.setMessageOplock("0");
			draft.setPatientIdentifier(AssigningAuthorityCode.SECURE_MESSAGE, userId);
			draft.setFrom("zztest patient01");
			draft.setFromId(userId);
			draft.setMailGroup("4004");
			draft.setTo("Patient Advocate - Test VAMC");
			draft.setToId("4004");
			messages.add(draft);
		}

		int deletedMessages = 30;
		SecureMessage deleted = new SecureMessage();
		for(int i = 0; i < deletedMessages; i++) {
			deleted = new SecureMessage();
			deleted.setAddresseeId(Integer.toString(addresseeId.addAndGet(1)));
			deleted.setAddresseeOplock("0");
			deleted.setThreadId(UUID.randomUUID().toString());
			deleted.setThreadOplock(UUID.randomUUID().toString());
			deleted.setDateSent(DateHelper.parseDate("06/"+Integer.toString(i)+"/2012"));
			deleted.setFolder("Deleted");
			deleted.setFolderId(SecureMessageCode.FOLDER_ID_DELETED.toString());
			deleted.setDataIdentifier("mock", "300"+Integer.toString(i));
			deleted.setMessageBody("Deleted Message "+Integer.toString(i));
			deleted.setMessageOplock("0");
			deleted.setPatientIdentifier(AssigningAuthorityCode.SECURE_MESSAGE, userId);
			deleted.setSubject("Test Result that I trashed");
			deleted.setSubjectCategory(SecureMessageCode.SUBJECT_CATEGORY_ID_TESTRESULT.toString());
			deleted.setFrom("My HealtheVet - Test VAMC");
			deleted.setFromId("4003");
			deleted.setMailGroup("4003");
			deleted.setTo("zztest patient01");
			deleted.setToId(userId);
			deleted.setReadDate(new Date());
			messages.add(deleted);
		}
		
        int pharmacyMessages = 5;
		SecureMessage pharmacy = new SecureMessage();
		for(int i = 0; i < pharmacyMessages; i++) {
			pharmacy = new SecureMessage();
			pharmacy.setAddresseeId(Integer.toString(addresseeId.addAndGet(1)));
			pharmacy.setAddresseeOplock("0");
			pharmacy.setThreadId(UUID.randomUUID().toString());
			pharmacy.setThreadOplock(UUID.randomUUID().toString());
			pharmacy.setDateSent(DateHelper.parseDate("06/"+Integer.toString(i)+"/2012"));
			pharmacy.setFolder("Pharmacy");
			pharmacy.setFolderId("500"+i);
			pharmacy.setDataIdentifier("mock", "300"+Integer.toString(i));
			pharmacy.setMessageBody("Pharmacy Message "+Integer.toString(i));
			pharmacy.setMessageOplock("0");
			pharmacy.setPatientIdentifier(AssigningAuthorityCode.SECURE_MESSAGE, userId);
			pharmacy.setSubject("Mock Data Item");
			pharmacy.setSubjectCategory(SecureMessageCode.SUBJECT_CATEGORY_ID_TESTRESULT.toString());
			pharmacy.setFrom("My HealtheVet - Test VAMC");
			pharmacy.setFromId("4003");
			pharmacy.setMailGroup("4003");
			pharmacy.setTo("zztest patient01");
			pharmacy.setToId(userId);
			pharmacy.setReadDate(new Date());
			messages.add(pharmacy);
		}
        
        int archiveMessages = 5;
		SecureMessage archive = new SecureMessage();
		for(int i = 0; i < archiveMessages; i++) {
			archive = new SecureMessage();
			archive.setAddresseeId(Integer.toString(addresseeId.addAndGet(1)));
			archive.setAddresseeOplock("0");
			archive.setThreadId(UUID.randomUUID().toString());
			archive.setThreadOplock(UUID.randomUUID().toString());
			archive.setDateSent(DateHelper.parseDate("06/"+Integer.toString(i)+"/2012"));
			archive.setFolder("Pharmacy");
			archive.setFolderId(Integer.toString(addresseeId.addAndGet(1)));
			archive.setDataIdentifier("mock", "300"+Integer.toString(i));
			archive.setMessageBody("Pharmacy Message "+Integer.toString(i));
			archive.setMessageOplock("0");
			archive.setPatientIdentifier(AssigningAuthorityCode.SECURE_MESSAGE, userId);
			archive.setSubject("Mock Data Item");
			archive.setSubjectCategory(SecureMessageCode.SUBJECT_CATEGORY_ID_TESTRESULT.toString());
			archive.setFrom("My HealtheVet - Test VAMC");
			archive.setFromId("4003");
			archive.setMailGroup("4003");
			archive.setTo("zztest patient01");
			archive.setToId(userId);
			archive.setReadDate(new Date());
			messages.add(archive);
		}

        
        
		SecureMessage sent = new SecureMessage();
		sent.setAddresseeId(Integer.toString(addresseeId.addAndGet(1)));
		sent.setAddresseeOplock("0");
		sent.setThreadId(UUID.randomUUID().toString());
		sent.setThreadOplock(UUID.randomUUID().toString());
		sent.setDateSent(DateHelper.parseDate("06/26/2012"));
		sent.setFolder(SecureMessageCode.FOLDER_NAME_SENT);
		sent.setFolderId(SecureMessageCode.FOLDER_ID_SENT.toString());
		sent.setDataIdentifier("mock", "1000");
		sent.setMessageBody("Sent Message 01");
		sent.setMessageOplock("0");
		sent.setPatientIdentifier(AssigningAuthorityCode.SECURE_MESSAGE, userId);
		sent.setSubject("Sent Subject - Appointment");
		sent.setSubjectCategory(SecureMessageCode.SUBJECT_CATEGORY_ID_APPOINTMENT.toString());
		sent.setTo("Patient Advocate - Test VAMC");
		sent.setToId("4004");
		sent.setFrom("zztest patient01");
		sent.setMailGroup("4004");
		sent.setFromId(userId);
		sent.setReadDate(new Date());
		messages.add(sent);

		messageStorage.put(userId, messages);
	}

	@Override
	public SecureMessageUser fetchSecureMessageUser(String userId) {
		return userStorage.get(userId);
	}

	@Override
	public SecureMessageFolders fetchFolders(String userId) {
		return folderStorage.get(userId);
	}

	@Override
	public SecureMessageContacts fetchContacts(String userId) {
		return contactStorage.get(userId);
	}

	@Override
	public SecureMessages fetchSecureMessages(String userId, SecureMessageFilter secureMessageFilter) {
		SecureMessages storedMessages = messageStorage.get(userId);

		SecureMessages filtered = new SecureMessages();
		for (SecureMessage message : storedMessages) {
			if (message.getFolderId().equals(secureMessageFilter.getFolder())) {
				filtered.add(message);
			}
		}
		
		Collections.sort(filtered);
		
		return processForReturn(filtered, secureMessageFilter);
	}

	@Override
	public SecureMessage readSecureMessage(String userId, SecureMessage secureMessage) {

		SecureMessage match = findMessage(userId, secureMessage.getUniqueId());
		if (!match.getAddresseeOplock().equals(secureMessage.getAddresseeOplock())) {
			throw new RuntimeException("You must have the latest version (same oplock) of a message to read it. Need to reload your data?");
		}

		match.setReadDate(new Date());
		Integer addresseeOplock = Integer.parseInt(match.getAddresseeOplock()) + 1;
		match.setAddresseeOplock(addresseeOplock.toString());

		return match;
	}

	@Override
	public SecureMessage sendSecureMessage(String userId, SecureMessage secureMessage) {

		setMessageIds(secureMessage);

		secureMessage.setFolder(SecureMessageCode.FOLDER_NAME_SENT);
		secureMessage.setFolderId(SecureMessageCode.FOLDER_ID_SENT.toString());
		secureMessage.setDateSent(new Date());
		addMessage(userId, secureMessage);

		return secureMessage;
	}

	@Override
	public SecureMessage replySecureMessage(String userId, SecureMessage secureMessage) {
		if (NullChecker.isNullish(secureMessage.getReplyToMessageId())) {
			throw new RuntimeException("replyToMessageId must contain the id of the contact that you are able to send to");
		}

		setMessageIds(secureMessage);
		secureMessage.setFolder(SecureMessageCode.FOLDER_NAME_SENT);
		secureMessage.setFolderId(SecureMessageCode.FOLDER_ID_SENT.toString());
		secureMessage.setDateSent(new Date());
		addMessage(userId, secureMessage);

		return secureMessage;
	}

	@Override
	public SecureMessage moveSecureMessage(String userId, String messageId, String folderId) {
		validateMoveLocation(folderId);
		SecureMessage messageToMove = findMessage(userId, messageId);

		return moveMessage(userId, folderId, messageToMove);
	}

	@Override
	public SecureMessage saveSecureMessageDraft(String userId, SecureMessage secureMessage) {

		if (NullChecker.isNullish(secureMessage.getUniqueId())) {
			setMessageIds(secureMessage);
			secureMessage.setFolder(SecureMessageCode.FOLDER_NAME_DRAFT);
			secureMessage.setFolderId(SecureMessageCode.FOLDER_ID_DRAFT.toString());
			secureMessage.setAddresseeId(Integer.toString(addresseeId.addAndGet(1)));
			addMessage(userId, secureMessage);

			return secureMessage;
		} else {
			// update existing - fake it out by removing it from the existing folder and adding it back
			return moveMessage(userId, secureMessage.getFolderId(), secureMessage);
		}
	}

	@Override
	public SecureMessage sendSecureMessageDraft(String userId, SecureMessage secureMessage) {
		logger.debug(String.format("Sending Draft with [id=%s]", secureMessage.getUniqueId()));
		secureMessage.setDateSent(new Date());
		
		return moveMessage(userId, SecureMessageCode.FOLDER_ID_SENT.toString(), secureMessage);
	}

	@Override
	public void deleteSecureMessageDraft(String userId, String messageId) {
		SecureMessages secureMessages = messageStorage.get(userId);

		SecureMessage toDelete = findMessage(userId, messageId);

		synchronized(this) {
			secureMessages.remove(toDelete);
		}
	}

	private void addMessage(String userId, SecureMessage secureMessage) {
		SecureMessages secureMessages = messageStorage.get(userId);
		if (secureMessages == null) {
			secureMessages = new SecureMessages();
			secureMessages.add(secureMessage);
			messageStorage.put(userId, secureMessages);
		} else {
			synchronized(this) {
				secureMessages.add(secureMessage);
			}
		}
	}

	private SecureMessage findMessage(String userId, String messageId) {
		SecureMessages secureMessages = messageStorage.get(userId);

		SecureMessage match = null;
		
		synchronized (this) {
			for (SecureMessage message : secureMessages) {
				System.out.println("Unique Id:" + message.getUniqueId());
				if (message.getUniqueId().equals(messageId)) {
					match = message;
				}
			}
		}

		return match;
	}

	private void validateMoveLocation(String folderId) {
		if (NullChecker.isNotNullish(folderId)) {
			if (folderId.equals(SecureMessageCode.FOLDER_ID_INBOX.toString()) || folderId.equals(SecureMessageCode.FOLDER_ID_DELETED.toString()) || (Integer.parseInt(folderId) > 0)) {
				return;
			} else {
				throw new RuntimeException("You can only move to Inbox, Deleted, or to custom folders");
			}
		} else {
			throw new RuntimeException("Folder id not provided during move");
		}
	}

	void changeFolderName(SecureMessage secureMessage) {
		if (secureMessage.getFolderId().equals(SecureMessageCode.FOLDER_ID_INBOX.toString())) {
			secureMessage.setFolder(SecureMessageCode.FOLDER_NAME_INBOX);
		} else if (secureMessage.getFolderId().equals(SecureMessageCode.FOLDER_ID_SENT.toString())) {
			secureMessage.setFolder(SecureMessageCode.FOLDER_NAME_SENT);
		} else if (secureMessage.getFolderId().equals(SecureMessageCode.FOLDER_ID_DRAFT.toString())) {
			secureMessage.setFolder(SecureMessageCode.FOLDER_NAME_DRAFT);
		} else if (secureMessage.getFolderId().equals(SecureMessageCode.FOLDER_ID_DELETED.toString())) {
			secureMessage.setFolder(SecureMessageCode.FOLDER_NAME_DELETED);
		} else if (secureMessage.getFolderId().equals(SecureMessageCode.FOLDER_ID_ESCALATED.toString())) {
			secureMessage.setFolder(SecureMessageCode.FOLDER_NAME_ESCALATED);
		} else if (secureMessage.getFolderId().equals(SecureMessageCode.FOLDER_ID_COMPLETED.toString())) {
			secureMessage.setFolder(SecureMessageCode.FOLDER_NAME_COMPLETED);
		} else if (secureMessage.getFolderId().equals(SecureMessageCode.FOLDER_ID_REMINDER.toString())) {
			secureMessage.setFolder(SecureMessageCode.FOLDER_NAME_REMINDER);
		} else {
			for (SecureMessageFolder folder : folderStorage.get(secureMessage.getPatientId())) {
				if (secureMessage.getFolderId().equals(folder.getUniqueId())) {
					secureMessage.setFolder(folder.getName());
				}
			}
		}
	}

	SecureMessage moveMessage(String userId, String folderId, SecureMessage newMessage) {
		deleteSecureMessage(userId, newMessage.getUniqueId());

		newMessage.setFolderId(folderId);
		changeFolderName(newMessage);
		addMessage(userId, newMessage);

		return newMessage;
	}

	synchronized void deleteSecureMessage(String userId, String messageId) {
		SecureMessages secureMessages = messageStorage.get(userId);
		int indexOfMessageToRemove = -1;
		if (NullChecker.isNotNullish(secureMessages)) {
			for(int i = 0; i < secureMessages.size(); i++) {
				SecureMessage message = secureMessages.get(i);
				if (NullSafeStringComparer.areEqual(message.getUniqueId(), messageId)) {
					indexOfMessageToRemove = i;
					break;
				}
			}
		}
		
		if (indexOfMessageToRemove != -1) {
			secureMessages.remove(indexOfMessageToRemove);
		}
	}

	void setMessageIds(SecureMessage secureMessage) {
		secureMessage.setAddresseeId(Integer.toString(addresseeId.addAndGet(1)));
		secureMessage.setAddresseeOplock("0");
		secureMessage.setThreadId(UUID.randomUUID().toString());
		secureMessage.setThreadOplock(UUID.randomUUID().toString());
		secureMessage.setMessageOplock("0");

		if (secureMessage.getUniqueId() == null) {
			secureMessage.setUniqueId(UUID.randomUUID().toString());
		}
	}

	SecureMessages processForReturn(SecureMessages secureMessages, SecureMessageFilter filter) {
		SecureMessages cleared = clearBodies(secureMessages);
		return paginate(cleared, filter);
	}

	SecureMessages paginate(SecureMessages secureMessages, SecureMessageFilter filter) {
		Paginator<SecureMessage, SecureMessages> paginator = new Paginator<SecureMessage, SecureMessages>(Integer.parseInt(filter.getPageNumber()), Integer.parseInt(filter.getPageSize()), 1);
		SecureMessages pageOfMessages = new SecureMessages();

		paginator.paginate(secureMessages, pageOfMessages);

		return pageOfMessages;
	}

	synchronized SecureMessages clearBodies(SecureMessages secureMessages) {
		SecureMessages cleanedMessages = new SecureMessages();

		for (SecureMessage original : secureMessages) {
			SecureMessage clone = new SecureMessage(original);
			clone.setMessageBody("");
			cleanedMessages.add(clone);
		}

		return cleanedMessages;
	}

}
