package com.agilex.healthcare.mobilehealthplatform.datalayer.patient.excel;

import java.io.IOException;
import java.io.InputStream;

import jxl.Cell;
import jxl.Sheet;
import jxl.Workbook;

import com.agilex.healthcare.mobilehealthplatform.domain.DomainTransferObjectCollection;

public abstract class ExcelCollectionParser<Tcollection extends DomainTransferObjectCollection<Titem>, Titem> {
	private static final int HEADER_ROW = 0;
	private static final org.apache.commons.logging.Log LOGGER = org.apache.commons.logging.LogFactory.getLog(ExcelCollectionParser.class);

	protected abstract Tcollection createNewCollection();

	protected abstract Titem createNewItem(Sheet sheet, int row);

	protected Workbook getWorkbook(String resourceName) {
		Workbook workbook = null;
		InputStream resourceStream = null;
		try {
			LOGGER.debug("Ready to open resource to read data.  Resource=" + resourceName);
			resourceStream = this.getClass().getResourceAsStream(resourceName);
			LOGGER.debug("Ready to open resource by stream.  " + resourceStream);
			workbook = Workbook.getWorkbook(resourceStream);
			LOGGER.debug("Excel workbook found and ready to be read.");
		} catch (Exception e) {
			// if no workbook found, then we will return no workbook and an
			// empty set of data
			LOGGER.debug("No Excel workbook found for " + resourceName);
		} finally {
			if (resourceStream != null) {
				try {
					resourceStream.close();
				} catch (IOException e) {
					LOGGER.error("Error closing workbook stream for " + resourceName, e);
				}
			}
		}

		return workbook;
	}

	public Tcollection getDataCollection(String resourceName) {
		Tcollection collection = createNewCollection();

		Workbook workbook = getWorkbook(resourceName);
		if (workbook != null) {
			int numberOfSheets = workbook.getNumberOfSheets();

			// for each sheet
			for (int sheetNumber = 0; sheetNumber < numberOfSheets; sheetNumber++)

			{
				Sheet sheet = workbook.getSheet(sheetNumber);
				LOGGER.debug("processing sheet " + sheet.getName());

				int numberOfColumns = sheet.getColumns();
				for (int column = 0; column < numberOfColumns; column++) {
					LOGGER.debug("header cell=" + getHeader(sheet, column));
				}

				LOGGER.debug("numberOfColumns=" + numberOfColumns);

				// for each row starting with second row
				int numberOfRows = sheet.getRows();
				for (int row = 1; row < numberOfRows; row++) {
					Titem item = createNewItem(sheet, row);
					collection.add(item);
				}
			}
			workbook.close();
		}
		return collection;
	}

	protected String getHeader(Sheet sheet, int column) {
		return getValue(sheet, column, HEADER_ROW);
	}

	protected String getValue(Sheet sheet, String columnName, int row) {
		Integer columnNumber = findColumnNumber(sheet, columnName);
		if (columnNumber == null)
			return null;
		return getValue(sheet, columnNumber, row);
	}

	private Integer findColumnNumber(Sheet sheet, String columnName) {
		int numberOfColumns = sheet.getColumns();
		for (int column = 0; column < numberOfColumns; column++) {
			String header = getHeader(sheet, column);
			LOGGER.debug("header cell=" + header);
			if (header.equalsIgnoreCase(columnName))
				return column;
		}
		return null;
	}

	protected String getValue(Sheet sheet, int column, int row) {
		Cell cell;
		try {
			cell = sheet.getCell(column, row);
		} catch (ArrayIndexOutOfBoundsException e) {
			throw new RuntimeException(String.format("Unable to read cell [%s=>%s,%s]", sheet.getName(), column, row), e);
		}
		String cellValue = cell.getContents();
		return cellValue;
	}

}
