package com.agilex.healthcare.mobilehealthplatform.datalayer.notification;

import java.util.List;

import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Response.Status;

import com.agilex.healthcare.mobilehealthplatform.MockDataStorage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.xls.DataElement;
import com.agilex.healthcare.mobilehealthplatform.datalayer.xls.DataElementDataStorage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.xls.DataElements;
import com.agilex.healthcare.mobilehealthplatform.domain.Notification;
import com.agilex.healthcare.mobilehealthplatform.domain.Notifications;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterer;
import com.agilex.healthcare.utility.NullSafeStringComparer;


public class NotificationDataLayerExcel implements NotificationDataLayer {

	private static final org.apache.commons.logging.Log LOGGER = org.apache.commons.logging.LogFactory.getLog(NotificationDataLayerExcel.class);

	private static MockDataStorage<Notifications, Notification> data = new MockDataStorage<Notifications, Notification>() {
		@Override
		protected Notifications createDataCollection() {
			return new Notifications();
		}
	};

	private static final String COLUMN_NOTIFICATION_ID = "NotificationId";
	private static final String COLUMN_PATIENTID = "UserId";
	private static final String COLUMN_DATE = "Date";
	private static final String COLUMN_BODY = "Body";
	private static final String COLUMN_ACTIVE_FLAG = "Active";
	private static final String DOMAIN = "Notifications";
	private static final int MAX_NUMBER_OF_DISPLAYED_NOTIFICATIONS = 50;
	
	@Override
	public Notifications getActiveNotifications(String userId, DateFilter filter) {
		LOGGER.debug("get notifications for patient " + userId);
		
		if (!data.dataExists(userId)) {
			DataElementDataStorage storage = DataElementDataStorage.getInstance();;
			DataElements dataElements = storage.getData(userId, DOMAIN);
			data.setDataElements(createNotifications(dataElements));
		}
		Notifications retrieved = data.getDataForPatient(userId);
		retrieved = sortAndTruncateNotifications(retrieved);
		
		return DateFilterer.filterByDate(retrieved, filter);
	}
	
	private Notifications sortAndTruncateNotifications(Notifications retrieved) {
		retrieved.sortDescending("date");
		Notifications returnedNotifications = new Notifications();
		
		if (retrieved.size() > MAX_NUMBER_OF_DISPLAYED_NOTIFICATIONS) {
			List<Notification> truncatedList = retrieved.subList(0, MAX_NUMBER_OF_DISPLAYED_NOTIFICATIONS);
			returnedNotifications.addAll(truncatedList);
		} else {
			returnedNotifications = retrieved;
		}
		
		return returnedNotifications;
	}

	@Override
	public Notification getNotification(String userId, String notificationId) {
		LOGGER.debug(String.format("get notification for notificationId [%s]", notificationId));
		
		Notification match = null;
		DateFilter emptyFilter = DateFilterFactory.createEmptyFilter();
		Notifications patientNotifications = getActiveNotifications(userId, emptyFilter);
		for (Notification potentialMatch : patientNotifications) {
			if (NullSafeStringComparer.areEqual(notificationId, potentialMatch.getUniqueId())) {
				match = potentialMatch;
				break;
			}
		}
		if (match == null) {
			throw new WebApplicationException(Status.NOT_FOUND);
		}
		return match;
	}
	
	@Override
	public void deleteNotification(Notification notification) {
		LOGGER.debug(String.format(":::: attempting to remove a notification with id %s", notification.getUniqueId()));
		Notification notificationToRemove = null;
		Notifications patientNotifications = data.getDataForPatient(notification.getPatientId());
		for (Notification potentialMatch : patientNotifications) {
			if (NullSafeStringComparer.areEqual(potentialMatch.getUniqueId(), notification.getUniqueId())) {
				LOGGER.debug(String.format(":::: Found a match %s", notification.getUniqueId()));
				notificationToRemove = potentialMatch;
				break;
			}
		}

		if (notificationToRemove != null) {
			LOGGER.debug(String.format(":::: Removing a notification with id %s", notificationToRemove.getUniqueId()));
			notificationToRemove.setActiveFlag(false);
			saveNotification(notificationToRemove);
			patientNotifications.remove(notificationToRemove);
		}
	}
	
	@Override
	public Notification saveNotification(Notification notification) {
		setNotification(notification);
		return notification;
	}

	private static synchronized void setNotification(Notification notification) {
		data.setDataElement(notification);
	}

	private Notifications createNotifications(DataElements dataElements) {
		Notifications notifications = new Notifications();

		for (DataElement dataElement : dataElements) {
			Notification notification = createNotification(dataElement);
			if (notification.isActiveFlag()) {
				notifications.add(notification);
			}
		}

		return notifications;
	}

	private Notification createNotification(DataElement dataElement) {
		Notification notification = new Notification();
		notification.setNotificationId(dataElement.getValue(COLUMN_NOTIFICATION_ID));
		notification.setPatientId(dataElement.getValue(COLUMN_PATIENTID));
		notification.setBody(dataElement.getValue(COLUMN_BODY));
		notification.setDate(dataElement.getValueAsDate(COLUMN_DATE));
		notification.setActiveFlag(dataElement.getValueAsBoolean(COLUMN_ACTIVE_FLAG));
		return notification;
	}
}
