package com.agilex.healthcare.mobilehealthplatform.datalayer.notification;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;

import com.agilex.healthcare.mobilehealthplatform.domain.DeviceRegistration;
import com.agilex.healthcare.mobilehealthplatform.domain.DeviceRegistrations;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.NullSafeStringComparer;

public class DeviceRegistrationDataLayerMock implements DeviceRegistrationDataLayer {
	private static final org.apache.commons.logging.Log LOGGER = org.apache.commons.logging.LogFactory.getLog(DeviceRegistrationDataLayerMock.class);

	private static Map<String, DeviceRegistrations> deviceRegistrationStorage = new ConcurrentHashMap<String, DeviceRegistrations>();

	static {
		LOGGER.debug("initialize DeviceRegistrationDataLayerMock storage");
		init("D123401");
	}

	private static synchronized void init(String userId) {
		DeviceRegistrations deviceRegistrations = new DeviceRegistrations();

		DeviceRegistration deviceRegistration = new DeviceRegistration();
		deviceRegistration.setUserId(userId);
		deviceRegistration.setDeviceToken("testDeviceToken");
		deviceRegistration.setRegisteredDate(new Date());
		deviceRegistration.setDeviceRegistrationId(UUID.randomUUID().toString());
		deviceRegistration.setFirstName("Gallow");
		deviceRegistration.setLastName("Younger");
		deviceRegistrations.add(deviceRegistration);

		deviceRegistrationStorage.put(userId, deviceRegistrations);
	}

	private static synchronized DeviceRegistration saveOrUpdateDeviceRegistration(DeviceRegistration registration) {
		DeviceRegistrations deviceRegistrations = deviceRegistrationStorage.get(registration.getUserId());
		LOGGER.info(deviceRegistrations);
		if (registration.getDeviceRegistrationId() == null) {
			registration.setDeviceRegistrationId(generateUniqueId());
		}

		if (deviceRegistrations == null) {
			deviceRegistrations = new DeviceRegistrations();
			deviceRegistrations.add(registration);
			deviceRegistrationStorage.put(registration.getUserId(), deviceRegistrations);
		} else {
			DeviceRegistration removeThis = null;
			for (DeviceRegistration replacementCandidate : deviceRegistrations) {
				if (replacementCandidate.getDeviceRegistrationId().equals(registration.getDeviceRegistrationId())) {
					removeThis = replacementCandidate;
					break;
				}
			}
			if (removeThis != null) {
				deviceRegistrations.remove(removeThis);
			}

			deviceRegistrations.add(registration);
		}

		return registration;
	}

	private static synchronized String generateUniqueId() {
		return UUID.randomUUID().toString();
	}

	public DeviceRegistrations getRegistrations() {
		DeviceRegistrations deviceRegistrations = new DeviceRegistrations();
		for (String key : deviceRegistrationStorage.keySet()) {
			deviceRegistrations.addAll(deviceRegistrationStorage.get(key));
		}
		return deviceRegistrations;
	}

	@Override
	public DeviceRegistration getRegistration(String userId, String deviceToken) {
		DeviceRegistrations deviceRegistrations = deviceRegistrationStorage.get(userId);

		DeviceRegistration deviceRegistrationToReturn = null;
		if (NullChecker.isNotNullish(deviceRegistrations)) {
			for (DeviceRegistration deviceRegistration : deviceRegistrations) {
				if (NullSafeStringComparer.areEqual(deviceToken, deviceRegistration.getDeviceToken())) {
					deviceRegistrationToReturn = deviceRegistration;
					break;
				}
			}
		}
		return deviceRegistrationToReturn;
	}

	@Override
	public DeviceRegistrations getRegistrations(String userId) {
		DeviceRegistrations registrations = deviceRegistrationStorage.get(userId);
		DeviceRegistrations activeRegistrations = new DeviceRegistrations();
		if (NullChecker.isNotNullish(registrations)) {
			for (DeviceRegistration registration : registrations) {
				if (registration.isOptedIn()) {
					activeRegistrations.add(registration);
				}
			}
		}
		return activeRegistrations;
	}

	@Override
	public DeviceRegistration saveRegistration(DeviceRegistration deviceRegistration) {
		saveOrUpdateDeviceRegistration(deviceRegistration);

		return deviceRegistration;
	}

	public void deleteRegistration(DeviceRegistration registration) {
		DeviceRegistrations deviceRegistrations = deviceRegistrationStorage.get(registration.getUserId());

		DeviceRegistration registrationToRemove = null;
		for (DeviceRegistration deviceRegistration : deviceRegistrations) {
			if (NullSafeStringComparer.areEqual(registration.getDeviceToken(), deviceRegistration.getDeviceToken())) {
				registrationToRemove = deviceRegistration;
			}
		}

		if (registrationToRemove != null) {
			deviceRegistrations.remove(registrationToRemove);
		}
	}

	public void deleteRegistration(String deviceToken) {
		DeviceRegistrations deviceRegistrationsToDelete = new DeviceRegistrations();
		for (DeviceRegistration deviceRegistration : getRegistrations()) {
			if (NullSafeStringComparer.areEqual(deviceRegistration.getDeviceToken(), deviceToken)) {
				deviceRegistrationsToDelete.add(deviceRegistration);
			}
		}
		for (DeviceRegistration deviceRegistrationToDelete : deviceRegistrationsToDelete) {
			deleteRegistration(deviceRegistrationToDelete);
		}
	}

	@Override
	public DeviceRegistrations fetchDeviceRegistrations(List<String> userIds) {
		DeviceRegistrations deviceRegistrations = new DeviceRegistrations();

		for (String userId : userIds) {
			deviceRegistrations.addAll(getRegistrations(userId));
		}

		return deviceRegistrations;
	}

	public void optOut(String userId) {
		DeviceRegistrations registrations = deviceRegistrationStorage.get(userId);
		for (DeviceRegistration registration : registrations) {
			registration.setOptedIn(false);
		}
	}

	@Override
	public void updateOptStatus(String userId, boolean status) {
		DeviceRegistrations registrations = deviceRegistrationStorage.get(userId);
		for (DeviceRegistration registration : registrations) {
			registration.setOptedIn(false);
		}
	}
	
	

}
