package com.agilex.healthcare.mobilehealthplatform.datalayer.medication;

import java.util.Calendar;
import java.util.Date;
import java.util.Random;

import com.agilex.healthcare.mobilehealthplatform.MockDataStorage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.xls.DataElement;
import com.agilex.healthcare.mobilehealthplatform.datalayer.xls.DataElementDataStorage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.xls.DataElements;
import com.agilex.healthcare.mobilehealthplatform.domain.Medication;
import com.agilex.healthcare.mobilehealthplatform.domain.Medications;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.utility.NullSafeStringComparer;

public class MedicationDataLayerExcel implements MedicationDataLayer {

	private static MockDataStorage<Medications, Medication> data = new MockDataStorage<Medications, Medication>() {
		@Override
		protected Medications createDataCollection() {
			return new Medications();
		}
	};

	private static final String COLUMN_ORDERNUM = "OrderNum";
	private static final String COLUMN_PATIENTID = "PatientID";
	private static final String COLUMN_ACTIVE = "Active";
	private static final String COLUMN_DRUGNAME = "DrugName";
	private static final String COLUMN_PHARM = "Pharmacy";
	private static final String COLUMN_CAT = "Category";
	private static final String COLUMN_STARTDATE = "StartDate";
	private static final String COLUMN_ENDDATE = "EndDate";
	private static final String COLUMN_MEDSOURCE = "MedSource";
	private static final String COLUMN_SOURCESYS = "SourceSys";
	private static final String COLUMN_DETAIL = "Detail";
	private static final String COLUMN_PRESCRIPTIONID = "PrescriptionID";
	private static final String COLUMN_STATUS = "Status";
	private static final String COLUMN_SIG = "Sig";
	private static final String COLUMN_LASTFILLEDDATE = "LastFilledDate";
	private static final String COLUMN_EXPIREDDATE = "ExpiredDate";
	private static final String COLUMN_ELIGIBLEFORREFILL = "EligibleForRefill";
	private static final String COLUMN_TOTALREFILLS = "TotalRefills";
	private static final String COLUMN_REFILLS = "Refills";
	private static final String COLUMN_REFILLSREMAINING = "RefillsRemaining";
	private static final String COLUMN_QUANTITY = "Quantity";
	private static final String COLUMN_DAYSSUPPLY = "DaysSupply";
	private static final String domain = "Medications";

	@Override
	public Medications getMedications(PatientIdentifier patientId, boolean activeOnly) {
		if (!data.dataExists(patientId.getUniqueId())) {
			DataElementDataStorage storage = DataElementDataStorage.getInstance();
			DataElements dataElements = storage.getData(patientId.getUniqueId(), domain);
			data.setDataElements(createMedications(dataElements, activeOnly));
		}
		return data.getDataForPatient(patientId.getUniqueId());
	}

	private Medications createMedications(DataElements dataElements, boolean activeOnly) {
		Medications medications = new Medications();
		for (DataElement dataElement : dataElements) {
			Medication m = createMedication(dataElement, false);
			if (!activeOnly || m.isActive())
				medications.add(m);
		}
		return medications;
	}

	private Medication createMedication(DataElement dataElement, boolean includeDetail) {
		Medication m = null;
		if (dataElement != null) {
			m = new Medication();
			m.setOrderNumber(dataElement.getValue(COLUMN_ORDERNUM));
			m.setPatientId(dataElement.getValue(COLUMN_PATIENTID));
			m.setActive(dataElement.getValueAsBoolean(COLUMN_ACTIVE));
			m.setDrugName(dataElement.getValue(COLUMN_DRUGNAME));
			m.setPharmacy(dataElement.getValue(COLUMN_PHARM));
			m.setCategory(dataElement.getValue(COLUMN_CAT));
			m.setStartDate(dataElement.getValueAsDate(COLUMN_STARTDATE));
			m.setEndDate(dataElement.getValueAsDate(COLUMN_ENDDATE));
			m.setMedicationSource(dataElement.getValue(COLUMN_MEDSOURCE));
			m.setSourceSystem(dataElement.getValue(COLUMN_SOURCESYS));
			m.setPrescriptionId(dataElement.getValue(COLUMN_PRESCRIPTIONID));
			m.setSig(dataElement.getValue(COLUMN_SIG));
			m.setLastFilledDate(dataElement.getValueAsDate(COLUMN_LASTFILLEDDATE));
			m.setExpiredDate(dataElement.getValueAsDate(COLUMN_EXPIREDDATE));
			m.setTotalRefills(dataElement.getValueAsInteger(COLUMN_TOTALREFILLS));
			m.setEligibleForRefill(dataElement.getValueAsBoolean(COLUMN_ELIGIBLEFORREFILL));
			m.setRefills(dataElement.getValue(COLUMN_REFILLS));
			m.setRefillsRemaining(dataElement.getValueAsInteger(COLUMN_REFILLSREMAINING));
			m.setQuantity(dataElement.getValue(COLUMN_QUANTITY));
			m.setDaysSupply(dataElement.getValue(COLUMN_DAYSSUPPLY));
			m.setUniqueId(m.getOrderNumber());
			m.setId(m.getOrderNumber());
			
			String status = dataElement.getValue(COLUMN_STATUS);
			m.setStatus(status);

			Date expiredDate = dataElement.getValueAsDate(COLUMN_EXPIREDDATE);
			if ("EXPIRED".equals(status) && expiredDate == null) {
				Random random = new Random();
				Calendar recentlyExpiredDate = Calendar.getInstance();
				recentlyExpiredDate.add(Calendar.DATE, (random.nextInt(80) + 1) * -1);
				m.setExpiredDate(recentlyExpiredDate.getTime());
			} else {
				m.setExpiredDate(expiredDate);
			}
			if (includeDetail) {
				m.setMedicationDetail(dataElement.getValue(COLUMN_DETAIL));
			}
		}
		return m;
	}

	@Override
	public Medication getMedicationLite(PatientIdentifier patientId, String orderNumber) {
		DataElementDataStorage storage = DataElementDataStorage.getInstance();;
		DataElement dataElement = storage.getData(patientId.getUniqueId(), domain, orderNumber, COLUMN_ORDERNUM);
		Medication medication = createMedication(dataElement, false);
		return medication;
	}

	@Override
	public Medication getMedicationDetail(PatientIdentifier patientId, String orderNumber) {
		DataElementDataStorage storage = DataElementDataStorage.getInstance();;
		DataElement dataElement = storage.getData(patientId.getUniqueId(), domain, orderNumber, COLUMN_ORDERNUM);
		return createMedication(dataElement, true);
	}

	@Override
	public Medications getMedications(PatientIdentifier patientId, MedicationFilter medicationFilter) {
		DataElementDataStorage storage = DataElementDataStorage.getInstance();;
		DataElements dataElements = storage.getData(patientId.getUniqueId(), domain);
		return createMedications(dataElements, false);
	}

	protected void updateMedication(Medication updatedMedication) {
		Medication medicationToUpdate = null;
		Medications medications = data.getDataForPatient(updatedMedication.getPatientId());
		for(Medication medication : medications) {
			if(NullSafeStringComparer.areEqual(medication.getOrderNumber(), updatedMedication.getOrderNumber())) {
				medicationToUpdate = medication;
			}
 		}
		
		if(medicationToUpdate != null) {
			medications.remove(medicationToUpdate);
			medications.add(updatedMedication);
		}
	}
}
