package com.agilex.healthcare.mobilehealthplatform.datalayer.lab;

import java.io.*;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

import jxl.*;

import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.*;
import com.agilex.healthcare.utility.DateHelper;

public class LabDataExcelRetriever {
	private static final org.apache.commons.logging.Log LOGGER = org.apache.commons.logging.LogFactory.getLog(LabDataExcelRetriever.class);
	private static Map<String, LabResults> cachedLabResultMap = new ConcurrentHashMap<String, LabResults>();
	private static Map<String, LabResults> cachedLabResultLiteMap = new ConcurrentHashMap<String, LabResults>();

	private static final int ACCESSION_COLUMN = 0;
	private static final int RESULTEDDATE_COLUMN = 1;
	private static final int LABVALUE_COLUMN = 2;
	private static final int REFHIGH_COLUMN = 3;
	private static final int REFLOW_COLUMN = 4;
	private static final int ABNORMAL_INDICATOR_COLUMN = 5;
	private static final int INDICATOR_COLUMN = 6;
	private static final int SPECIMENNAME_COLUMN = 7;
	private static final int SPECIMENID_COLUMN = 8;
	private static final int ORDERID_COLUMN = 9;
	private static final int TESTID_COLUMN = 10;
	private static final int LOINC_COLUMN = 11;
	private static final int VALUEUNITS_COLUMN = 12;
	private static final int LABTYPE_COLUMN = 13;
	private static final int XML_OUTPUT = 14;
	private static final int DESCRIPTION_COLUMN = 15;

	private static final int TOPDATAROW = 1;

	public LabResults getLabResults(PatientIdentifier patientIdentifier, DateFilter dateFilter) {
		String patientId = patientIdentifier.getUniqueId();
		LabResults cachedLabResultsForThisPatient = null;// cachedLabResultMap.get(patientId);

		if (cachedLabResultsForThisPatient == null) {
			cachedLabResultsForThisPatient = getLabResultsForPatient(patientId, false);
			cachedLabResultMap.put(patientId, cachedLabResultsForThisPatient);
			LOGGER.debug("caching results for next run");
		} else {
			LOGGER.debug("returning results from cache");
		}

		LabResults filteredResults = DateFilterer.filterByDate(cachedLabResultsForThisPatient, dateFilter);

		return filteredResults;
	}

	public LabResults getLabResultsLite(String patientId, DateFilter dateFilter) {
		LOGGER.debug("::: Getting Lite Lab Results");
		LabResults cachedLabResults = null;// cachedLabResultLiteMap.get(patientId);

		if (cachedLabResults == null) {
			cachedLabResults = getLabResultsForPatient(patientId, true);
			cachedLabResultLiteMap.put(patientId, cachedLabResults);
			LOGGER.debug("caching results for next run");
		} else {
			LOGGER.debug("returning results from cache");
		}

		LabResults filteredResults = DateFilterer.filterByDate(cachedLabResults, dateFilter);

		return filteredResults;
	}

	private String getResourceName(String patientId) {
		return String.format("labresults.%s.xls", patientId);
	}

	private LabResults getLabResultsForPatient(String patientId, boolean lite) {
		LabResults labResults = new LabResults();

		Workbook workbook = getWorkbook(patientId);
		if (workbook != null) {
			int numberOfSheets = workbook.getNumberOfSheets();

			// each sheet represents a single test (with many results)
			for (int sheetNumber = 0; sheetNumber < numberOfSheets; sheetNumber++) {
				LabResults labResultsForThisTest = readLabTestFromSheet(workbook, sheetNumber, patientId, lite);
				labResults.addAll(labResultsForThisTest);
				LOGGER.debug(String.format("read %s labs from sheet %s for patientId %s, current total count %s", labResultsForThisTest.size(), sheetNumber, patientId, labResults.size()));
			}
			workbook.close();
		}
		return labResults;

	}

	private LabResults readLabTestFromSheet(Workbook workbook, int sheetNumber, String patientId, boolean lite) {
		LabResults labResults = new LabResults();
		String sheetName;
		Sheet singleTestSheet = workbook.getSheet(sheetNumber);
		sheetName = singleTestSheet.getName();

		int row = 1;

		while (thereIsMoreData(singleTestSheet, row)) {
			LabResult r = readLabResultFromRow(row, patientId, sheetName, singleTestSheet, lite);
			labResults.add(r);
			row++;
		}
		return labResults;
	}

	private LabResult readLabResultFromRow(int currentRow, String patientId, String sheetName, Sheet singleTestSheet, boolean lite) {
		LabResult r = new LabResult();

		r.setPatientId(patientId);
		r.setTestname(sheetName);
		r.setUniqueId(getValue(ACCESSION_COLUMN, currentRow, singleTestSheet));
		r.setTestId(getValue(TESTID_COLUMN, TOPDATAROW, singleTestSheet));
		r.setReferenceHigh(getValue(REFHIGH_COLUMN, currentRow, singleTestSheet));
		r.setReferenceLow(getValue(REFLOW_COLUMN, currentRow, singleTestSheet));
		r.setValueUnits(getValue(VALUEUNITS_COLUMN, TOPDATAROW, singleTestSheet));
		r.setAccessionNumber(getValue(ACCESSION_COLUMN, currentRow, singleTestSheet));
		r.setIndicator(getValue(INDICATOR_COLUMN, currentRow, singleTestSheet));
		r.setResultedDate(getValueAsDate(RESULTEDDATE_COLUMN, currentRow, singleTestSheet));
		r.setSpecimenId(getValue(SPECIMENID_COLUMN, TOPDATAROW, singleTestSheet));
		r.setSpecimenName(getValue(SPECIMENNAME_COLUMN, TOPDATAROW, singleTestSheet));
		r.setOrderId(getValue(ORDERID_COLUMN, currentRow, singleTestSheet));
		r.setLabType(getValue(LABTYPE_COLUMN, TOPDATAROW, singleTestSheet));
		r.setDisplayDescription(getValue(DESCRIPTION_COLUMN, currentRow, singleTestSheet));
		r.setLoinc(getValue(LOINC_COLUMN, currentRow, singleTestSheet));
		if (!lite)
			r.setValue(getValue(LABVALUE_COLUMN, currentRow, singleTestSheet));

		return r;
	}

	private boolean thereIsMoreData(Sheet sheet, int row) {
		Cell valueCell = sheet.getCell(LABVALUE_COLUMN, row);
		return valueCell.getContents() != null && !valueCell.getContents().isEmpty();
	}

	private String getValue(int column, int row, Sheet sheetNeeded) {
		return sheetNeeded.getCell(column, row).getContents();
	}

	private Date getValueAsDate(int column, int row, Sheet sheetNeeded) {
		return DateHelper.parseDate(getValue(column, row, sheetNeeded));
	}

	private Workbook getWorkbook(String patientId) {
		Workbook workbook = null;
		InputStream resourceStream = null;
		try {
			String resourceName = getResourceName(patientId);
			LOGGER.debug("Ready to open resource to read lab data.  Resource=" + resourceName);
			resourceStream = this.getClass().getResourceAsStream(resourceName);
			if (resourceStream == null)
				LOGGER.debug("Ready to open resource by stream.  " + resourceStream);

			WorkbookSettings ws = new WorkbookSettings();
			ws.setGCDisabled(true);
			workbook = Workbook.getWorkbook(resourceStream, ws);
		} catch (Exception e) {
			// if no workbook found, then we will return no workbook and an
			// empty set of labs
			LOGGER.error("Error loading workbook for " + patientId, e);
		} finally {
			if (resourceStream != null) {
				try {
					resourceStream.close();
				} catch (IOException e) {
					LOGGER.error("Error closing workbook stream for " + patientId, e);
				}
			}
		}

		LOGGER.debug("Ready to read lab data from resource.");
		return workbook;

	}

}
