package com.agilex.healthcare.mobilehealthplatform.datalayer.event;

import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import com.agilex.healthcare.mobilehealthplatform.datalayer.mood.MoodEventDataLayer;
import com.agilex.healthcare.mobilehealthplatform.domain.MoodEvent;
import com.agilex.healthcare.mobilehealthplatform.domain.MoodEvents;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterer;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.NullSafeStringComparer;

public class MoodEventDataLayerMock implements MoodEventDataLayer {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(MoodEventDataLayerMock.class);

	private static Map<String, MoodEvents> mockDataStorage = new HashMap<String, MoodEvents>();

	static {
		logger.debug("initialize MoodEventDataLayerMock storage");
		init("D123401");
		init("PATID26");
	}

	private static synchronized void init(String patientId) {
		MoodEvents events = new MoodEvents();

		MoodEvent event = new MoodEvent();
		event.setPatientId(patientId);
		event.setDate(DateHelper.parseDate("1/1/2012"));
		event.setUniqueId(generateId());
		event.setValue("1");
		event.setNotes("note1");
		events.add(event);

		event = new MoodEvent();
		event.setPatientId(patientId);
		event.setDate(DateHelper.parseDate("1/2/2012"));
		event.setUniqueId(generateId());
		event.setValue("2");
		events.add(event);

		event = new MoodEvent();
		event.setPatientId(patientId);
		event.setDate(DateHelper.parseDate("1/3/2012"));
		event.setUniqueId(generateId());
		event.setValue("3");
		events.add(event);

		event = new MoodEvent();
		event.setPatientId(patientId);
		event.setDate(DateHelper.parseDate("1/4/2012"));
		event.setUniqueId(generateId());
		event.setValue("4");
		events.add(event);

		event = new MoodEvent();
		event.setPatientId(patientId);
		event.setDate(DateHelper.parseDate("1/10/2012"));
		event.setUniqueId(generateId());
		event.setValue("10");
		events.add(event);

		mockDataStorage.put(patientId, events);
	}

	private static MoodEvent insertOrUpdateEvent(MoodEvent event) {
		MoodEvents patientMoods = mockDataStorage.get(event.getPatientId());
		if (event.getUniqueId() == null) {
			event.setUniqueId(generateId());
		}

		logger.debug(String.format("Saving Mood Event [patientId:%s][id:%s][value:%s]", event.getPatientId(), event.getUniqueId(), event.getValue()));

		if (patientMoods == null) {
			patientMoods = new MoodEvents();
			patientMoods.add(event);
			mockDataStorage.put(event.getPatientId(), patientMoods);
		} else {
			MoodEvent removeThis = null;
			for (MoodEvent replacementCandidate : patientMoods) {
				if (replacementCandidate.getUniqueId().equals(event.getUniqueId())) {
					removeThis = replacementCandidate;
					break;
				}
			}
			if (removeThis != null) {
				patientMoods.remove(removeThis);
			}
			patientMoods.add(event);
		}

		return event;
	}

	private static String generateId() {
		return UUID.randomUUID().toString();
	}

	@Override
	public MoodEvents getMoodEvents(String patientId) {
		return mockDataStorage.get(patientId);
	}

	@Override
	public MoodEvents getMoodEvents(String patientId, DateFilter filter) {
		MoodEvents moodEvents = mockDataStorage.get(patientId);

		MoodEvents filteredList = DateFilterer.filterByDate(moodEvents, filter);
		return filteredList;
	}

	@Override
	public MoodEvent getMoodEvent(String patientId, String eventId) {
		MoodEvents moodEvents = mockDataStorage.get(patientId);
		for (MoodEvent moodEvent : moodEvents) {
			if (moodEvent.getUniqueId().equals(eventId)) {
				return moodEvent;
			}
		}

		return null;
	}

	@Override
	public MoodEvent saveMoodEvent(MoodEvent moodEvent) {
		logger.debug(String.format("Saving Mood Event [id:%s][value:%s]", moodEvent.getUniqueId(), moodEvent.getValue()));
		return insertOrUpdateEvent(moodEvent);
	}

	@Override
	public void deleteMoodEvent(MoodEvent moodEvent) {
		logger.debug(String.format(":::: looking to remove mood id %s for patient %s", moodEvent.getUniqueId(), moodEvent.getPatientId()));
		MoodEvents moodEvents = mockDataStorage.get(moodEvent.getPatientId());
		MoodEvent eventToRemove = null;
		if (NullChecker.isNotNullish(moodEvents)) {
			logger.debug("Mood Events exist for the patient id");
			for (MoodEvent event : moodEvents) {
				if (NullSafeStringComparer.areEqual(event.getUniqueId(), moodEvent.getUniqueId()))
					eventToRemove = moodEvent;
			}
		}

		if (eventToRemove != null) {
			Boolean isRemoved = moodEvents.remove(eventToRemove);
			logger.debug(String.format(":::: removing mood id %s for patient %s ; success: %s", eventToRemove.getUniqueId(), eventToRemove.getPatientId(), isRemoved.toString()));
		}
	}

}
