package com.agilex.healthcare.mobilehealthplatform.datalayer.event;

import com.agilex.healthcare.mobilehealthplatform.MockDataStorage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.exercise.ExerciseDataLayer;
import com.agilex.healthcare.mobilehealthplatform.datalayer.exercise.ExerciseFilter;
import com.agilex.healthcare.mobilehealthplatform.datalayer.xls.DataElement;
import com.agilex.healthcare.mobilehealthplatform.datalayer.xls.DataElementDataStorage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.xls.DataElements;
import com.agilex.healthcare.mobilehealthplatform.domain.Exercise;
import com.agilex.healthcare.mobilehealthplatform.domain.Exercises;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterer;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.NullSafeStringComparer;

public class ExerciseDataLayerExcel implements ExerciseDataLayer {

	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(ExerciseDataLayerExcel.class);

	private static MockDataStorage<Exercises, Exercise> data = new MockDataStorage<Exercises, Exercise>() {
		@Override
		protected Exercises createDataCollection() {
			return new Exercises();
		}
	};

	private static final String COLUMN_ID = "Id";
	private static final String COLUMN_PATIENTID = "PatientId";
	private static final String COLUMN_ACTIVITY = "Activity";
	private static final String COLUMN_DURATION = "Duration";
	private static final String COLUMN_DISTANCE = "Distance";
	private static final String COLUMN_DATE = "Date";
	private static final String COLUMN_NOTES = "Notes";
	private static final String domain = "Exercise";

	@Override
	public Exercises getExercises(String patientId) {
		logger.debug("get exercise for patient " + patientId);
		if (!data.dataExists(patientId)) {
			DataElementDataStorage storage = DataElementDataStorage.getInstance();
			DataElements dataElements = storage.getData(patientId, domain);
			data.setDataElements(createExercises(dataElements));
		}
		return data.getDataForPatient(patientId);
	}

	@Override
	public Exercises getExercises(String patientId, DateFilter filter) {
		if (filter == null) {
			filter = new ExerciseFilter();
		}

		Exercises exercises = getExercises(patientId);
		if (NullChecker.isNullish(exercises)) {
			exercises = new Exercises();
		}
		
		Exercises filteredExercises = DateFilterer.filterByDate(exercises, filter);
		return filteredExercises;
	}

	@Override
	public Exercise getExercise(String patientId, String exerciseId) {
		Exercise match = null;
		Exercises patientExercises = getExercises(patientId);
		for (Exercise potentialMatch : patientExercises) {
			if (NullSafeStringComparer.areEqual(exerciseId, potentialMatch.getUniqueId())) {
				match = potentialMatch;
				break;
			}
		}
		return match;
	}

	@Override
	public Exercise saveExercise(Exercise exercise) {
		setExercise(exercise);
		return exercise;
	}

	@Override
	public void deleteExercise(Exercise exercise) {
		logger.debug(String.format(":::: attempting to remove an exercise with id %s", exercise.getUniqueId()));
		Exercise exerciseToRemove = null;
		Exercises patientExercises = data.getDataForPatient(exercise.getPatientId());
		for (Exercise potentialMatch : patientExercises) {
			if (NullSafeStringComparer.areEqual(potentialMatch.getUniqueId(), exercise.getUniqueId())) {
				logger.debug(String.format(":::: Found a match %s", exercise.getUniqueId()));
				exerciseToRemove = potentialMatch;
				break;
			}
		}

		if (exerciseToRemove != null) {
			logger.debug(String.format(":::: Removing an exercise with id %s", exerciseToRemove.getUniqueId()));
			patientExercises.remove(exerciseToRemove);
		}
	}

	private static synchronized void setExercise(Exercise exercise) {
		data.setDataElement(exercise);
	}

	private Exercises createExercises(DataElements dataElements) {
		Exercises exercises = new Exercises();

		for (DataElement dataElement : dataElements) {
			Exercise exercise = createExercise(dataElement);
			exercises.add(exercise);
		}

		return exercises;
	}

	private Exercise createExercise(DataElement dataElement) {
		Exercise exercise = new Exercise();

		exercise.setUniqueId(dataElement.getValue(COLUMN_ID));
		exercise.setPatientId(dataElement.getValue(COLUMN_PATIENTID));
		exercise.setActivity(dataElement.getValue(COLUMN_ACTIVITY));
		exercise.setDuration(Integer.parseInt(dataElement.getValue(COLUMN_DURATION)));
		exercise.setDistance(Double.parseDouble(dataElement.getValue(COLUMN_DISTANCE)));
		exercise.setDate(dataElement.getValueAsDate(COLUMN_DATE));
		exercise.setNotes(dataElement.getValue(COLUMN_NOTES));

		return exercise;
	}

}
