package com.agilex.healthcare.mobilehealthplatform.datalayer.event;

import com.agilex.healthcare.mobilehealthplatform.MockDataStorage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.dailyevents.DailyEventDataLayer;
import com.agilex.healthcare.mobilehealthplatform.datalayer.xls.DataElement;
import com.agilex.healthcare.mobilehealthplatform.datalayer.xls.DataElementDataStorage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.xls.DataElements;
import com.agilex.healthcare.mobilehealthplatform.domain.DailyEvent;
import com.agilex.healthcare.mobilehealthplatform.domain.DailyEvents;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterer;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.NullSafeStringComparer;

public class DailyEventDataLayerExcel implements DailyEventDataLayer {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(DailyEventDataLayerExcel.class);

	private static MockDataStorage<DailyEvents, DailyEvent> data = new MockDataStorage<DailyEvents, DailyEvent>() {
		@Override
		protected DailyEvents createDataCollection() {
			return new DailyEvents();
		}
	};

	private static final String COLUMN_PATIENTID = "PatientId";
	private static final String COLUMN_EVENTID = "EventId";
	private static final String COLUMN_NOTES = "Notes";
	private static final String COLUMN_ENTRYDATE = "EntryDate";
	private static final String COLUMN_TITLE = "Title";
	private static final String domain = "DailyEvents";

	private static synchronized void setDailyEvent(DailyEvent dailyEvent) {
		data.setDataElement(dailyEvent);
	}

	@Override
	public DailyEvents getDailyEvents(String patientId) {
		logger.debug("get daily events for patient " + patientId);
		if (!data.dataExists(patientId)) {
			DataElementDataStorage storage = DataElementDataStorage.getInstance();
			DataElements dataElements = storage.getData(patientId, domain);
			data.setDataElements(createDailyEvents(dataElements));
		}
		return data.getDataForPatient(patientId);
	}

	@Override
	public DailyEvents getDailyEvents(String patientId, DateFilter filter) {
		logger.debug("get daily events with filter for patient " + patientId);

		if (filter == null)
			filter = DateFilterFactory.createEmptyFilter();
		logger.debug("about to get daily events with filter for patient " + patientId);

		DailyEvents dailyEventsForPatient = getDailyEvents(patientId);

		if (dailyEventsForPatient == null) {
			dailyEventsForPatient = new DailyEvents();
		}
		logger.debug("daily events found : " + dailyEventsForPatient.size());

		DailyEvents filteredEvents = DateFilterer.filterByDate(dailyEventsForPatient, filter);
		logger.debug("daily events remaining : " + filteredEvents.size());
		return filteredEvents;
	}

	private DailyEvents createDailyEvents(DataElements dataElements) {
		DailyEvents dailyEvents = new DailyEvents();

		for (DataElement dataElement : dataElements) {
			DailyEvent dailyEvent = createDailyEvent(dataElement);
			dailyEvents.add(dailyEvent);
		}

		return dailyEvents;
	}

	private DailyEvent createDailyEvent(DataElement dataElement) {
		DailyEvent dailyEvent = new DailyEvent();

		dailyEvent.setPatientId(dataElement.getValue(COLUMN_PATIENTID));
		dailyEvent.setId(dataElement.getValue(COLUMN_EVENTID));
		dailyEvent.setNotes(dataElement.getValue(COLUMN_NOTES));
		dailyEvent.setEntryDate(dataElement.getValueAsDate(COLUMN_ENTRYDATE));
		dailyEvent.setTitle(dataElement.getValue(COLUMN_TITLE));

		return dailyEvent;

	}

	@Override
	public DailyEvent saveDailyEvent(DailyEvent dailyEvent) {
		setDailyEvent(dailyEvent);
		return dailyEvent;
	}

	@Override
	public DailyEvent getDailyEvent(String patientId, String eventId) {
		logger.debug("get daily event for patient " + patientId + " and event " + eventId);
		DailyEvent match = null;
		DailyEvents dailyEventsForPatient = getDailyEvents(patientId);
		logger.debug("total event for patient = " + dailyEventsForPatient);
		for (DailyEvent potentialMatch : dailyEventsForPatient) {
			logger.debug(String.format("compare %s==%s", eventId, potentialMatch.getId()));
			if (NullSafeStringComparer.areEqual(eventId, potentialMatch.getId())) {
				match = potentialMatch;
				break;
			}
		}
		return match;
	}

	@Override
	public void deleteDailyEvent(DailyEvent dailyEvent) {
		if (dailyEvent == null)
			throw new IllegalArgumentException("Daily event unexpectedly null, cannot delete");
		if (NullChecker.isNullish(dailyEvent.getPatientId()))
			throw new IllegalArgumentException("Daily event does not contain a patient identifier, cannot delete");
		if (NullChecker.isNullish(dailyEvent.getId()))
			throw new IllegalArgumentException("Daily event does not contain an identifier, cannot delete");

		DailyEvent existingEvent = null;
		DailyEvents dailyEvents = data.getDataForPatient(dailyEvent.getPatientId());
		for (DailyEvent potentialMatch : dailyEvents) {
			if (NullSafeStringComparer.areEqual(potentialMatch.getId(), dailyEvent.getId())) {
				existingEvent = potentialMatch;
				break;
			}
		}

		if (existingEvent != null) {
			dailyEvents.remove(existingEvent);
		}
	}

}
