package com.agilex.healthcare.mobilehealthplatform.datalayer.document;

import java.util.UUID;

import com.agilex.healthcare.mobilehealthplatform.domain.Document;
import com.agilex.healthcare.mobilehealthplatform.domain.Documents;
import com.agilex.healthcare.mobilehealthplatform.domain.RelatedDataItem;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.NullSafeStringComparer;

/**
 * 
 * @author j.ray
 */
public class DocumentDataLayerMock implements DocumentDataLayer {

	private Documents cachedDocuments = new Documents();
	
	@Override
	public Documents fetchDocuments(String patientIdentifier, DocumentFilter filter) {
		Documents documents = fetchDocumentsByPatient(patientIdentifier);
		documents = filterDocuments(filter, documents);
		return documents;
	}

	private Documents filterDocuments(DocumentFilter filter, Documents documents) {
		Documents matchedDocuments = documents;
		if (filter != null && filter.getParentItem() != null) {
			matchedDocuments = new Documents();
			for (Document potentialMatch : documents) {
				if (areParentItemsEqual(filter.getParentItem(), potentialMatch.getParentData())) {
					matchedDocuments.add(potentialMatch);
				}
			}
		}
		return matchedDocuments;
	}

	private boolean areParentItemsEqual(RelatedDataItem filterRelatedDataItem, RelatedDataItem parentData) {
		if (isValidRelatedItem(filterRelatedDataItem)) {
			return true;
		}
		if (isValidRelatedItem(parentData)) {
			return false;
		}
		if (!parentData.getDomain().contentEquals(parentData.getDomain())) {
			return false;
		}
		if (!parentData.getIdentifier().getUniqueId().contentEquals(filterRelatedDataItem.getIdentifier().getUniqueId())) {
			return false;
		}
		return true;
	}

	private boolean isValidRelatedItem(RelatedDataItem relatedItem) {
		return relatedItem == null || NullChecker.isNullish(relatedItem.getDomain()) || relatedItem.getIdentifier() == null || NullChecker.isNullish(relatedItem.getIdentifier().getUniqueId());
	}

	@Override
	public Document fetchDocument(String patientIdentifier, String identifier) {
		Document document = findByIdentifier(identifier);
		return document;
	}

	@Override
	public Document saveDocument(Document document) {
		if (NullChecker.isNullish(document.getUniqueId())) {
			document.setUniqueId(UUID.randomUUID().toString());
		}

		Document existingDocument = findByIdentifier(document.getUniqueId());
		if (existingDocument != null) {
			deleteDocument(existingDocument);
		}
		
		cachedDocuments.add(document);
		return document;
	}


	@Override
	public void deleteDocument(Document document) {
		Document documentToRemove = null;
		for(Document potentialMatch : cachedDocuments) {
			if(NullSafeStringComparer.areEqual(potentialMatch.getUniqueId(), document.getUniqueId())) {
				documentToRemove = potentialMatch;
				break;
			}
		}
		
		if(documentToRemove != null) {
			cachedDocuments.remove(documentToRemove);
		}
	}

	
	private Documents fetchDocumentsByPatient(String patientIdentifier) {
		Documents documentsForPatient = new Documents();
		Documents allDocuments = getDocumentList(false);

		for (Document document : allDocuments) {
			if (NullSafeStringComparer.areEqual(document.getPatientId(), patientIdentifier)) {
				documentsForPatient.add(document);
			}
		}

		return documentsForPatient;
	}

	private Documents getDocumentList(boolean includeDocumentBody) {
		return cachedDocuments;
	}
	

	private Document findByIdentifier(String identifier) {
		Documents allDocuments = getDocumentList(true);
		Document document = null;
		for (Document potentialMatch : allDocuments) {
			if (NullSafeStringComparer.areEqual(identifier, potentialMatch.getUniqueId())) {
				document = potentialMatch;
				break;
			}
		}
		return document;
	}

}
