package com.agilex.healthcare.mobilehealthplatform.datalayer.diet;

import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;

import com.agilex.healthcare.mobilehealthplatform.domain.DietEntries;
import com.agilex.healthcare.mobilehealthplatform.domain.DietEntry;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterer;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.MealType;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.NullSafeStringComparer;

public class DietDataLayerMock implements DietDataLayer {

	private static Map<String, DietEntries> mockDietDataStorage;
	static Integer mockGeneratedEntryId = 0;

	static {
		initializeStorage();
	}

	private static synchronized void initializeStorage() {
		mockDietDataStorage = new ConcurrentHashMap<String, DietEntries>();

		String gallowPatientId = "PATID26";

		DietEntry mealEntry = new DietEntry();
		mealEntry.setMealType(MealType.BREAKFAST);
		mealEntry.setId(UUID.randomUUID().toString());
		mealEntry.setEntryDate(DateHelper.parseDate("12/1/2011"));
		mealEntry.setFat("3");
		mealEntry.setProtein("8");
		mealEntry.setCalories("250");
		mealEntry.setCarbs("15");
		mealEntry.setNotes("An Omlet");
		mealEntry.setPatientId(gallowPatientId);
		setDietEntry(mealEntry);

		mealEntry = new DietEntry();
		mealEntry.setMealType(MealType.LUNCH);
		mealEntry.setId(UUID.randomUUID().toString());
		mealEntry.setEntryDate(DateHelper.parseDate("12/1/2011"));
		mealEntry.setFat("1");
		mealEntry.setProtein("4");
		mealEntry.setCalories("100");
		mealEntry.setCarbs("9");
		mealEntry.setNotes("Potato Chips");
		mealEntry.setPatientId(gallowPatientId);
		setDietEntry(mealEntry);

		mealEntry = new DietEntry();
		mealEntry.setMealType(MealType.LUNCH);
		mealEntry.setId(UUID.randomUUID().toString());
		mealEntry.setEntryDate(DateHelper.parseDate("12/1/2011"));
		mealEntry.setFat("3");
		mealEntry.setProtein("2");
		mealEntry.setCalories("160");
		mealEntry.setCarbs("10");
		mealEntry.setNotes("Turkey Sandwich");
		mealEntry.setPatientId(gallowPatientId);
		setDietEntry(mealEntry);

		mealEntry = new DietEntry();
		mealEntry.setMealType(MealType.LUNCH);
		mealEntry.setId(UUID.randomUUID().toString());
		mealEntry.setEntryDate(DateHelper.parseDate("12/1/2011"));
		mealEntry.setFat("0");
		mealEntry.setProtein("3");
		mealEntry.setCalories("100");
		mealEntry.setCarbs("20");
		mealEntry.setNotes("Apple");
		mealEntry.setPatientId(gallowPatientId);
		setDietEntry(mealEntry);

		mealEntry = new DietEntry();
		mealEntry.setMealType(MealType.DINNER);
		mealEntry.setId(UUID.randomUUID().toString());
		mealEntry.setEntryDate(DateHelper.parseDate("12/1/2011"));
		mealEntry.setFat("1");
		mealEntry.setProtein("4");
		mealEntry.setCalories("200");
		mealEntry.setCarbs("0");
		mealEntry.setNotes("Broccoli");
		mealEntry.setPatientId(gallowPatientId);
		setDietEntry(mealEntry);

		mealEntry = new DietEntry();
		mealEntry.setMealType(MealType.DINNER);
		mealEntry.setId(UUID.randomUUID().toString());
		mealEntry.setEntryDate(DateHelper.parseDate("12/1/2011"));
		mealEntry.setFat("6");
		mealEntry.setProtein("12");
		mealEntry.setCalories("300");
		mealEntry.setCarbs("21");
		mealEntry.setNotes("Meatloaf");
		mealEntry.setPatientId(gallowPatientId);
		setDietEntry(mealEntry);

		mealEntry = new DietEntry();
		mealEntry.setMealType(MealType.SNACK);
		mealEntry.setId(UUID.randomUUID().toString());
		mealEntry.setEntryDate(DateHelper.parseDate("12/1/2011"));
		mealEntry.setFat("4");
		mealEntry.setProtein("0");
		mealEntry.setCalories("220");
		mealEntry.setCarbs("7");
		mealEntry.setNotes("Cookies");
		mealEntry.setPatientId(gallowPatientId);
		setDietEntry(mealEntry);

		mealEntry = new DietEntry();
		mealEntry.setMealType(MealType.DINNER);
		mealEntry.setId(UUID.randomUUID().toString());
		mealEntry.setEntryDate(DateHelper.parseDate("12/2/2011"));
		mealEntry.setFat("21");
		mealEntry.setProtein("18");
		mealEntry.setCalories("600");
		mealEntry.setCarbs("21");
		mealEntry.setNotes("Fried Chicken");
		mealEntry.setPatientId(gallowPatientId);
		setDietEntry(mealEntry);

		mealEntry = new DietEntry();
		mealEntry.setMealType(MealType.SNACK);
		mealEntry.setId(UUID.randomUUID().toString());
		mealEntry.setEntryDate(DateHelper.parseDate("12/2/2011"));
		mealEntry.setFat("5.5");
		mealEntry.setProtein("1");
		mealEntry.setCalories("320");
		mealEntry.setCarbs("5");
		mealEntry.setNotes("Cake");
		mealEntry.setPatientId(gallowPatientId);
		setDietEntry(mealEntry);

		mealEntry = new DietEntry();
		mealEntry.setMealType(MealType.BREAKFAST);
		mealEntry.setId(UUID.randomUUID().toString());
		mealEntry.setEntryDate(DateHelper.parseDate("12/1/2011"));
		mealEntry.setFat("1");
		mealEntry.setProtein("2");
		mealEntry.setCalories("100");
		mealEntry.setCarbs("5");
		mealEntry.setNotes("food");
		mealEntry.setPatientId("D123401");
		setDietEntry(mealEntry);
	}

	private static synchronized void setDietEntry(DietEntry mealEntry) {
		if (NullChecker.isNullish(mealEntry.getId())) {
			generateAndPopulateIdentifier(mealEntry);
		}

		DietEntries mealEntries = mockDietDataStorage.get(mealEntry.getPatientId());

		if (mealEntries == null) {
			mealEntries = new DietEntries();
			mockDietDataStorage.put(mealEntry.getPatientId(), mealEntries);
		}

		DietEntry existingMeal = null;
		for (DietEntry potentialMatch : mealEntries) {
			if (NullSafeStringComparer.areEqual(potentialMatch.getId(), mealEntry.getId())) {
				existingMeal = potentialMatch;
				break;
			}
		}

		if (existingMeal != null) {
			mealEntries.remove(existingMeal);
		}

		mealEntries.add(mealEntry);
	}

	@Override
	public DietEntries getDietEntries(String patientId) {
		return getDietEntries(patientId, null);
	}

	@Override
	public DietEntries getDietEntries(String patientId, DateFilter filter) {
		if (filter == null)
			filter = DateFilterFactory.createEmptyFilter();


		DietEntries mealEntriesForPatient = mockDietDataStorage.get(patientId);
		if (mealEntriesForPatient == null) {
			mealEntriesForPatient = new DietEntries();
		}

		DietEntries filteredList = DateFilterer.filterByDate(mealEntriesForPatient, filter);

		return filteredList;
	}

	@Override
	public DietEntry saveDietEntry(DietEntry mealEntry) {
		setDietEntry(mealEntry);
		return mealEntry;
	}

	private static void generateAndPopulateIdentifier(DietEntry mealEntry) {
		mealEntry.setId((++mockGeneratedEntryId).toString());
	}

	@Override
	public DietEntry getDietEntry(String patientId, String mealId) {
		DietEntry match = null;
		DietEntries mealEntriesForPatient = getDietEntries(patientId);
		for (DietEntry potentialMatch : mealEntriesForPatient) {
			if (NullSafeStringComparer.areEqual(mealId, potentialMatch.getId()))
				match = potentialMatch;
			break;
		}
		return match;
	}

	@Override
	public void deleteDietEntry(DietEntry mealEntry) {
		if (mealEntry == null)
			throw new IllegalArgumentException("Meal entry unexpectedly null, cannot delete");
		if (NullChecker.isNullish(mealEntry.getPatientId()))
			throw new IllegalArgumentException("Meal entry does not contain a patient identifier, cannot delete");
		if (NullChecker.isNullish(mealEntry.getId()))
			throw new IllegalArgumentException("Meal entry does not contain an identifier, cannot delete");

		DietEntry existingMeal = null;
		DietEntries mealEntries = mockDietDataStorage.get(mealEntry.getPatientId());
		for (DietEntry potentialMatch : mealEntries) {
			if (NullSafeStringComparer.areEqual(potentialMatch.getId(), mealEntry.getId())) {
				existingMeal = potentialMatch;
				break;
			}
		}

		if (existingMeal != null) {
			mealEntries.remove(existingMeal);
		}
	}

}
