package com.agilex.healthcare.mobilehealthplatform.datalayer.diet;

import com.agilex.healthcare.mobilehealthplatform.MockDataStorage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.xls.DataElement;
import com.agilex.healthcare.mobilehealthplatform.datalayer.xls.DataElementDataStorage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.xls.DataElements;
import com.agilex.healthcare.mobilehealthplatform.domain.DietEntries;
import com.agilex.healthcare.mobilehealthplatform.domain.DietEntry;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterer;
import com.agilex.healthcare.utility.MealType;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.NullSafeStringComparer;

public class DietDataLayerExcel implements DietDataLayer {
	private static MockDataStorage<DietEntries, DietEntry> data = new MockDataStorage<DietEntries, DietEntry>() {
		@Override
		protected DietEntries createDataCollection() {
			return new DietEntries();
		}
	};

	private static final String COLUMN_PATIENTID = "PatientID";
	private static final String COLUMN_UNIQUEID = "UniqueID";
	private static final String COLUMN_MEALTYPE = "MealType";
	private static final String COLUMN_ENTRYDATE = "EntryDate";
	private static final String COLUMN_FAT = "Fat";
	private static final String COLUMN_PROTEIN = "Protein";
	private static final String COLUMN_CALORIES = "Calories";
	private static final String COLUMN_CARBS = "Carbs";
	private static final String COLUMN_NOTES = "Notes";
	private static final String domain = "Diet";

	@Override
	public DietEntries getDietEntries(String patientId) {
		if(!data.dataExists(patientId)) {
			DataElementDataStorage storage = DataElementDataStorage.getInstance();
			DataElements dataElements = storage.getData(patientId, domain);
			data.setDataElements(createDietEntries(dataElements));
		}
	return data.getDataForPatient(patientId);
	}

	@Override
	public DietEntries getDietEntries(String patientId, DateFilter filter) {
		DietEntries dietEntries = getDietEntries(patientId);
		if(NullChecker.isNullish(dietEntries)) {
			dietEntries = new DietEntries();
		}
		DietEntries filteredDietEntries = DateFilterer.filterByDate(dietEntries, filter);
		return filteredDietEntries;
	}

	@Override
	public DietEntry getDietEntry(String patientId, String dietEntryId) {
		DietEntry match = null;
		DietEntries patientDietEntries = getDietEntries(patientId);
		for (DietEntry potentialMatch : patientDietEntries) {
			if(NullSafeStringComparer.areEqual(dietEntryId, potentialMatch.getUniqueId())) {
				match = potentialMatch;
				break;
			}				
		}
		return match;
	}

	@Override
	public DietEntry saveDietEntry(DietEntry mealEntry) {
		setDietEntry(mealEntry);
		return mealEntry;
	}

	@Override
	public void deleteDietEntry(DietEntry dietEntry) {
		DietEntry dietEntryToRemove = null;
		DietEntries patientDietEntries = data.getDataForPatient(dietEntry.getPatientId());
		for (DietEntry potentialMatch : patientDietEntries) {
			if(NullSafeStringComparer.areEqual(potentialMatch.getUniqueId(), dietEntry.getUniqueId())) {
				dietEntryToRemove = potentialMatch;
				break;
			}
		}
		if(dietEntryToRemove != null) {
			patientDietEntries.remove(dietEntryToRemove);
		}
	}

	private static synchronized void setDietEntry(DietEntry dietEntry) {
		data.setDataElement(dietEntry);		
	}

	private DietEntries createDietEntries(DataElements dataElements) {
		DietEntries dietEntries = new DietEntries();
		for(DataElement dataElement : dataElements) {
			DietEntry dietEntry = createDietEntry(dataElement);
			dietEntries.add(dietEntry);
		}
		return dietEntries;
	}	

	private DietEntry createDietEntry(DataElement dataElement) {
		DietEntry dietEntry = new DietEntry();
		dietEntry.setPatientId(dataElement.getValue(COLUMN_PATIENTID));
		dietEntry.setUniqueId(dataElement.getValue(COLUMN_UNIQUEID));		
		MealType mealType = MealType.fromValue(dataElement.getValue(COLUMN_MEALTYPE));
		dietEntry.setMealType(mealType);
		dietEntry.setEntryDate(dataElement.getValueAsDate(COLUMN_ENTRYDATE));
		dietEntry.setFat(dataElement.getValue(COLUMN_FAT));
		dietEntry.setProtein(dataElement.getValue(COLUMN_PROTEIN));
		dietEntry.setCalories(dataElement.getValue(COLUMN_CALORIES));
		dietEntry.setCarbs(dataElement.getValue(COLUMN_CARBS));
		dietEntry.setNotes(dataElement.getValue(COLUMN_NOTES));		
		return dietEntry;
	}
}
