package com.agilex.healthcare.mobilehealthplatform.datalayer.contactlog;

import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;

import com.agilex.healthcare.mobilehealthplatform.domain.ContactLog;
import com.agilex.healthcare.mobilehealthplatform.domain.ContactLogs;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterer;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.NullSafeStringComparer;

public class ContactLogDataLayerMock implements ContactLogDataLayer {
	private static final org.apache.commons.logging.Log LOGGER = org.apache.commons.logging.LogFactory.getLog(ContactLogDataLayerMock.class);

	private static Map<String, ContactLogs> mockDataStorage;
	static String mockGeneratedId;

	static {
		initializeStorage("D123401"); // gallow
		initializeStorage("PATID26"); // aldie
		initializeStorage("01");
		initializeStorage("02");
		initializeStorage("03");
		initializeStorage("04");
		initializeStorage("05");
		initializeStorage("06");
		initializeStorage("07");
		initializeStorage("08");
		initializeStorage("09");
		initializeStorage("10");
	}

	public static synchronized void initializeStorage(String patientId) {
		LOGGER.debug("initializing contactLog Storage");
		mockDataStorage = new ConcurrentHashMap<String, ContactLogs>();

		ContactLog log = new ContactLog();
		log.setActivity("Phone Call");
		log.setContact("Dr. Fitness");
		log.setLogTime(DateHelper.parseDate("3/1/2012"));
		log.setNote("Be More Active");
		log.setPatientId(patientId);
		log.setTask("Morning Jog");
		saveOrUpdateContactLog(log);

		log = new ContactLog();
		log.setActivity("Email");
		log.setContact("Dr. Medication");
		log.setLogTime(DateHelper.parseDate("3/2/2012"));
		log.setNote("Requested a new prescription for Allergy Medications");
		log.setPatientId(patientId);
		log.setTask("Follow-up Tomorrow");
		saveOrUpdateContactLog(log);

		log = new ContactLog();
		log.setActivity("Phone Call");
		log.setContact("Dr. Medical Test");
		log.setLogTime(DateHelper.parseDate("3/3/2012"));
		log.setNote("Requested an MRI Rx");
		log.setPatientId(patientId);
		log.setTask("CT scheduled for March 5");
		saveOrUpdateContactLog(log);

		log = new ContactLog();
		log.setActivity("Phone Call");
		log.setContact("Pharmacist");
		log.setLogTime(DateHelper.parseDate("3/4/2012"));
		log.setNote("Rx is now available");
		log.setPatientId(patientId);
		log.setTask("Pick up Rx");
		saveOrUpdateContactLog(log);

		log = new ContactLog();
		log.setActivity("Phone Call");
		log.setContact("Dr. Longwinder");
		log.setLogTime(DateHelper.parseDate("3/5/2012"));
		log.setNote("Long note long notelong notelong notelong notelong notelong notelong notelong notelong notelong notelong note"
				+ "long notelong notelong notelong notelong notelong notelong notelong notelong notelong notelong notelong notelong note"
				+ "long notelong notelong notelong notelong notelong notelong notelong notelong notelong notelong notelong notelong note"
				+ "long notelong notelong notelong notelong notelong notelong notelong notelong notelong notelong notelong notelong note");
		log.setPatientId(patientId);
		log.setTask("Test how long notes look on the device");
		saveOrUpdateContactLog(log);
	}

	private static synchronized ContactLog saveOrUpdateContactLog(ContactLog contactLog) {
		ContactLogs patientLogs = mockDataStorage.get(contactLog.getPatientId());
		LOGGER.info(patientLogs);
		if (contactLog.getId() == null) {
			contactLog.setId(generateUniqueId());
		}

		if (patientLogs == null) {
			patientLogs = new ContactLogs();
			patientLogs.add(contactLog);
			mockDataStorage.put(contactLog.getPatientId(), patientLogs);
		} else {
			ContactLog removeThis = null;
			for (ContactLog replacementCandidate : patientLogs) {
				if (replacementCandidate.getId().equals(contactLog.getId())) {
					removeThis = replacementCandidate;
					break;
				}
			}
			if (removeThis != null) {
				patientLogs.remove(removeThis);
			}

			patientLogs.add(contactLog);

		}

		return contactLog;
	}

	private static synchronized String generateUniqueId() {
		return UUID.randomUUID().toString();
	}

	@Override
	public ContactLog getContactLog(String patientId, String contactLogId) {
		ContactLogs patientLogs = mockDataStorage.get(patientId);
		ContactLog match = null;
		if (patientLogs != null) {
			LOGGER.info("mock data storage found the following number of objects: " + patientLogs.size());
			for (ContactLog contactLog : patientLogs) {
				if (NullSafeStringComparer.areEqual(contactLogId, contactLog.getId())) {
					match = contactLog;
					LOGGER.info("match was found with id " + match.getId() + " " + match.getPatientId());
				}
			}
		} else {
			LOGGER.info("mock data storage returned null");
		}

		return match;
	}

	@Override
	public ContactLogs getContactLogs(String patientId) {
		ContactLogs logs = mockDataStorage.get(patientId);
		return logs;
	}

	@Override
	public ContactLogs getContactLogs(String patientId, DateFilter filter) {
		ContactLogs contactLogs = mockDataStorage.get(patientId);
		if (contactLogs == null)
			contactLogs = new ContactLogs();
		LOGGER.debug("The number of unfiltered logs returned is " + contactLogs.size());
		ContactLogs filteredList = DateFilterer.filterByDate(contactLogs, filter);
		LOGGER.debug("The number of filtered logs returned is " + filteredList.size());

		return filteredList;
	}

	@Override
	public ContactLog saveContactLog(ContactLog contactLog) {
		LOGGER.info("Patient Id passed is " + contactLog.getPatientId());

		saveOrUpdateContactLog(contactLog);

		LOGGER.info("Patient id after save is " + contactLog.getPatientId());
		return contactLog;
	}

	@Override
	public void deleteContactLog(ContactLog contactLog) {
		LOGGER.info("Patient for delete is" + contactLog.getPatientId());
		ContactLogs logs = mockDataStorage.get(contactLog.getPatientId());
		ContactLog logToRemove = null;
		if (NullChecker.isNotNullish(logs)) {
			for (ContactLog log : logs) {
				if (NullSafeStringComparer.areEqual(log.getId(), contactLog.getId())) {
					logToRemove = log;
				}
			}
		}

		if (logToRemove != null) {
			logs.remove(logToRemove);
		}
	}
}
