package com.agilex.healthcare.mobilehealthplatform.datalayer.contactlog;

import com.agilex.healthcare.mobilehealthplatform.MockDataStorage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.xls.DataElement;
import com.agilex.healthcare.mobilehealthplatform.datalayer.xls.DataElementDataStorage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.xls.DataElements;
import com.agilex.healthcare.mobilehealthplatform.domain.ContactLog;
import com.agilex.healthcare.mobilehealthplatform.domain.ContactLogs;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterer;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.NullSafeStringComparer;

public class ContactLogDataLayerExcel implements ContactLogDataLayer {

	private static MockDataStorage<ContactLogs, ContactLog> data = new MockDataStorage<ContactLogs, ContactLog>() {
		@Override
		protected ContactLogs createDataCollection() {
			return new ContactLogs();
		}
	};

	private static final String COLUMN_CONTACTLOGID = "ContactLogID";
	private static final String COLUMN_ACTIVITY = "Activity";
	private static final String COLUMN_CONTACT = "Contact";
	private static final String COLUMN_LOGTIME = "LogTime";
	private static final String COLUMN_NOTE = "Note";
	private static final String COLUMN_TASK = "Task";
	private static final String COLUMN_PATIENTID = "PatientID";
	private static final String COLUMN_SOURCESYSTEM = "SourceSystem";
	private static final String domain = "ContactLogs";

	@Override
	public ContactLogs getContactLogs(String patientId) {
		if (!data.dataExists(patientId)) {
			DataElementDataStorage storage = DataElementDataStorage.getInstance();
			DataElements dataElements = storage.getData(patientId, domain);
			data.setDataElements(createContactLogs(dataElements));
		}
		return data.getDataForPatient(patientId);
	}
	
	@Override
	public ContactLogs getContactLogs(String patientId, DateFilter filter) {
		ContactLogs contactLogs = getContactLogs(patientId);
		if (NullChecker.isNullish(contactLogs)) {
			contactLogs = new ContactLogs();
		}
		ContactLogs filteredContactLogs = DateFilterer.filterByDate(contactLogs, filter);
		return filteredContactLogs;
	}

	
	@Override
	public ContactLog getContactLog(String patientId, String contactLogId) {
		ContactLog match = null;
		ContactLogs patientContactLogs = getContactLogs(patientId);
		for(ContactLog potentialMatch : patientContactLogs) {
			if(NullSafeStringComparer.areEqual(contactLogId, potentialMatch.getUniqueId())) {
				match = potentialMatch;
				break;
			}
		}
		return match;
	}


	@Override
	public ContactLog saveContactLog(ContactLog contactLog) {
		setContactLog(contactLog);
		return contactLog;
	}

	@Override
	public void deleteContactLog(ContactLog contactLog) {
		ContactLog contactLogToRemove = null;
		ContactLogs patientContactLogs = getContactLogs(contactLog.getPatientId());
		for (ContactLog potentialMatch : patientContactLogs) {
			if (NullSafeStringComparer.areEqual(potentialMatch.getUniqueId(), contactLog.getId())) {
				contactLogToRemove = potentialMatch;
				break;
			}
		}

		if (contactLogToRemove != null) {
			patientContactLogs.remove(contactLogToRemove);
		}
	}

	private static synchronized void setContactLog(ContactLog contactLog) {
		data.setDataElement(contactLog);
	}

	private ContactLogs createContactLogs(DataElements dataElements) {
		ContactLogs contactLogs = new ContactLogs();
		for (DataElement dataElement : dataElements) {
			ContactLog contactLog = createContactLog(dataElement);
			contactLogs.add(contactLog);
		}
		return contactLogs;
	}

	private ContactLog createContactLog(DataElement dataElement) {
		ContactLog contactLog = null;
		if (dataElement != null) {
			contactLog = new ContactLog();
			contactLog.setId(dataElement.getValue(COLUMN_CONTACTLOGID));
			contactLog.setActivity(dataElement.getValue(COLUMN_ACTIVITY));
			contactLog.setContact(dataElement.getValue(COLUMN_CONTACT));
			contactLog.setLogTime(dataElement.getValueAsDate(COLUMN_LOGTIME));
			contactLog.setNote(dataElement.getValue(COLUMN_NOTE));
			contactLog.setTask(dataElement.getValue(COLUMN_TASK));
			contactLog.setPatientId(dataElement.getValue(COLUMN_PATIENTID));
			contactLog.setSourceSystem(dataElement.getValue(COLUMN_SOURCESYSTEM));
		}
		return contactLog;
	}
}
