package com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresults;

import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResult;
import com.agilex.healthcare.mobilehealthplatform.domain.AssessmentResults;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterer;
import com.agilex.healthcare.utility.NullChecker;

public class AssessmentResultsDataLayerMock implements AssessmentResultDataLayer {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(AssessmentResultsDataLayerMock.class);
	private static Map<String, AssessmentResults> mockDataStorage = new HashMap<String, AssessmentResults>();

	@Override
	public AssessmentResult fetchAssessmentResult(PatientIdentifier patientIdentifier, String resultId) {
		AssessmentResult retrievedAssessmentResult = null;
		AssessmentResults resultsForPatientId = mockDataStorage.get(patientIdentifier.getUniqueId());
		for (AssessmentResult assessmentResult : resultsForPatientId) {
			if (assessmentResult.getUniqueId().equals(resultId)) {
				retrievedAssessmentResult = assessmentResult;
			}
		}
		return retrievedAssessmentResult;
	}

	@Override
	public AssessmentResults fetchResultsByAssessment(PatientIdentifier patientIdentifier, String assessmentId, DateFilter filter) {
		if (NullChecker.isNullish(assessmentId))
			throw new IllegalArgumentException("assessmentId required");

		AssessmentResults resultsForPatient = mockDataStorage.get(patientIdentifier.getUniqueId());

		if (resultsForPatient == null)
			resultsForPatient = new AssessmentResults();

		AssessmentResults fetchedResults = new AssessmentResults();
		for (AssessmentResult assessmentResult : resultsForPatient) {
			if (assessmentResult.getAssessmentId().equals(assessmentId) && !assessmentResult.isInProgress()) {
				fetchedResults.add(assessmentResult);
			}
		}
		return DateFilterer.filterByDate(fetchedResults, filter);
	}

	@Override
	public AssessmentResults fetchAssessmentDraftsByPatient(PatientIdentifier patientIdentifier) {
		logger.debug(String.format("fetching assessment drafts by patient [patient=%s]", patientIdentifier));
		AssessmentResults assessmentResults = fetchAssessmentResultsByPatient(patientIdentifier.getUniqueId());
		AssessmentResults drafts = filterDrafts(assessmentResults);
		for (AssessmentResult draft : drafts) {
			logger.debug(String.format("draft [%s]", draft.getAssessmentId()));
		}
		logger.debug(String.format("fetch assessment drafts by patient [patient=%s] returned %s row(s)", patientIdentifier, drafts.size()));
		return drafts;
	}

	@Override
	public AssessmentResult saveAssessmentResult(AssessmentResult assessmentResult) {
		assessmentResult.setUniqueId(generateRandomResultId());
		String patientId = assessmentResult.getPatientId();
		AssessmentResults existingAssessmentResults = fetchAssessmentResultsByPatient(patientId);
		existingAssessmentResults = removePotentialExistingDraft(assessmentResult, existingAssessmentResults);
		existingAssessmentResults.add(assessmentResult);
		mockDataStorage.put(patientId, existingAssessmentResults);
		return assessmentResult;
	}

	private String generateRandomResultId() {
		String randomResultId = UUID.randomUUID().toString();
		return randomResultId;
	}

	@Override
	public AssessmentResult fetchMostRecentlyCompletedResult(String assessmentId, PatientIdentifier patientIdentifier) {
		AssessmentResults allPreviousResults = fetchPreviouslyCompletedResults(assessmentId, patientIdentifier.getUniqueId());
		AssessmentResult mostRecentCompletedResult = determineMostRecentlyCompleted(allPreviousResults);
		return mostRecentCompletedResult;
	}

	private AssessmentResult determineMostRecentlyCompleted(AssessmentResults allPreviousResults) {
		AssessmentResult mostRecentCompletedResult = null;
		if (allPreviousResults.size() > 0) {
			for (AssessmentResult assessmentResult : allPreviousResults) {
				if (mostRecentCompletedResult == null) {
					mostRecentCompletedResult = assessmentResult;
				} else if (mostRecentCompletedResult.getDateTaken().before(assessmentResult.getDateTaken())) {
					mostRecentCompletedResult = assessmentResult;
				}
			}
		}
		return mostRecentCompletedResult;
	}

	private AssessmentResults fetchPreviouslyCompletedResults(String assessmentId, String patientId) {
		AssessmentResults allResultsForPatient = fetchAssessmentResultsByPatient(patientId);
		AssessmentResults previouslyCompletedResults = new AssessmentResults();
		for (AssessmentResult assessmentResult : allResultsForPatient) {
			if (!assessmentResult.isInProgress() && assessmentResult.getAssessmentId().equals(assessmentId)) {
				previouslyCompletedResults.add(assessmentResult);
			}
		}
		return previouslyCompletedResults;
	}

	public AssessmentResults fetchAssessmentResultsByPatient(String patientId) {
		AssessmentResults resultsForPatient = mockDataStorage.get(patientId);
		if (resultsForPatient == null) {
			resultsForPatient = new AssessmentResults();
		}
		return resultsForPatient;
	}

	private AssessmentResults removePotentialExistingDraft(AssessmentResult assessmentResult, AssessmentResults updatedAssessmentResults) {
		String assessmentId = assessmentResult.getAssessmentId();
		AssessmentResult resultToRemove = updatedAssessmentResults.getResultByAssessmentId(assessmentId);
		if (resultToRemove != null && resultToRemove.isInProgress()) {
			updatedAssessmentResults.remove(resultToRemove);
		}
		return updatedAssessmentResults;
	}

	private AssessmentResults filterDrafts(AssessmentResults allAssessmentResultsForPatient) {
		AssessmentResults drafts = new AssessmentResults();
		for (AssessmentResult assessmentResult : allAssessmentResultsForPatient) {
			if (assessmentResult.isInProgress()) {
				drafts.add(assessmentResult);
			}
		}
		return drafts;
	}
}