package com.agilex.healthcare.mobilehealthplatform.datalayer.appointment;

import java.util.Calendar;
import java.util.Random;

import com.agilex.healthcare.mobilehealthplatform.datalayer.xls.DataElement;
import com.agilex.healthcare.mobilehealthplatform.datalayer.xls.DataElementDataStorage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.xls.DataElements;
import com.agilex.healthcare.mobilehealthplatform.domain.Appointment;
import com.agilex.healthcare.mobilehealthplatform.domain.Appointments;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterer;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.NullSafeStringComparer;

public class AppointmentsDataLayerExcel implements AppointmentDataLayer {
	private static final String domain = "appointments";
	private static final String COLUMN_UNIQUEID = "UniqueId";
	private static final String COLUMN_STARTDATE = "StartDate";
	private static final String COLUMN_ENDDATE = "EndDate";
	private static final String COLUMN_CLINICID = "ClinicId";
	private static final String COLUMN_CLINICNAME = "ClinicName";
	private static final String COLUMN_CLINICTYPE = "ClinicType";
	private static final String COLUMN_FACILITYNAME = "Facility";
	private static final String COLUMN_PATIENTSSN = "PatientSsn";
	private static final String COLUMN_PROVIDERID = "ProviderId";
	private static final String COLUMN_PROVIDERNAME = "ProviderName";
	private static final String COLUMN_REASONFORVISIT = "RFV";
	private static final String COLUMN_STATUS = "Status";
	private static final String COLUMN_PATIENTID = "PatientId";
	private static final String COLUMN_CLINIC_VISITID = "ClinicalVisitId";
	private static final String COLUMN_CLINICALNOTES = "ClinicalNotes";
	private static final String COLUMN_CLINICPHONE = "ClinicPhone";

	private static Appointments allAppointments;

	@Override
	public Appointments getProviderAppointments(String providerId, DateFilter filter) {
		return applyFilter(getAllAppointments(), filter, null, providerId);
	}

	@Override
	public Appointments getClinicAppointments(String clinicId, DateFilter filter) {
		return applyFilter(getAllAppointments(), filter, clinicId, null);
	}

	private synchronized Appointments getAllAppointments() {
		if (allAppointments == null) {
			Appointments appointments = new Appointments();
			DataElementDataStorage storage = DataElementDataStorage.getInstance();;
			String[] knownPatients = { "a", "D123401", "PATID22", "PATID40" };
			for (String patientId : knownPatients) {
				DataElements dataElements = storage.getData(patientId, domain);
				Appointments appointmentsForThisPatient = createAppointments(dataElements, false);
				appointments.addAll(appointmentsForThisPatient);
			}
			allAppointments = appointments;
		}
		return allAppointments;
	}

    @Override
    public Appointments fetchAppointmentsByPatient(PatientIdentifier patientIdentifier, DateFilter filter) {
        return fetchAppointmentsByPatient(patientIdentifier, filter, false);
    }

    @Override
    public Appointments fetchAppointmentsByPatient(PatientIdentifier patientIdentifier, DateFilter filter, boolean shouldIncludeDetails) {
		DataElementDataStorage storage = DataElementDataStorage.getInstance();
		DataElements dataElements = storage.getData(patientIdentifier.getUniqueId(), domain);
		Appointments appointments = createAppointments(dataElements, shouldIncludeDetails);
		return DateFilterer.filterByDate(appointments, filter);
    }

    private Appointments createAppointments(DataElements dataElements, boolean includeClinicNote) {
		Appointments appointments = new Appointments();
		for (DataElement dataElement : dataElements) {
			appointments.add(createAppointment(dataElement, includeClinicNote));
		}
		return appointments;
	}

	private Appointment createAppointment(DataElement dataElement, boolean includeClinicalNotes) {
		Appointment a = new Appointment();
		a.setUniqueId(dataElement.getValue(COLUMN_UNIQUEID));
		a.setPatientId(dataElement.getValue(COLUMN_PATIENTID));
		a.setClinicId(dataElement.getValue(COLUMN_CLINICID));
		a.setClinicName(dataElement.getValue(COLUMN_CLINICNAME));
		a.setClinicType(dataElement.getValue(COLUMN_CLINICTYPE));
		a.setFacilityName(dataElement.getValue(COLUMN_FACILITYNAME));
		a.setPatientSSN(dataElement.getValue(COLUMN_PATIENTSSN));
		a.setProviderId(dataElement.getValue(COLUMN_PROVIDERID));
		a.setProviderName(dataElement.getValue(COLUMN_PROVIDERNAME));
		a.setReasonForVisit(dataElement.getValue(COLUMN_REASONFORVISIT));
		a.setStatus(dataElement.getValue(COLUMN_STATUS));
		a.setClinicPhone(dataElement.getValue(COLUMN_CLINICPHONE));
		a.setClinicVisitId(dataElement.getValue(COLUMN_CLINIC_VISITID));
		if (includeClinicalNotes) {
			a.setClinicalNotes(dataElement.getValue(COLUMN_CLINICALNOTES));
		}
		setDates(a, dataElement);

		return a;
	}
	
	private void setDates(Appointment appointment, DataElement dataElement) {
		if (dataElement.getValueAsDateTime(COLUMN_STARTDATE) == null || dataElement.getValueAsDateTime(COLUMN_ENDDATE) == null) {
			Random random = new Random();
			Calendar date = Calendar.getInstance();
			date.add(Calendar.DATE, random.nextInt(25) + 1);
			appointment.setAppointmentStartDate(date.getTime());
			date.add(Calendar.DATE, 1);
			appointment.setAppointmentEndDate(date.getTime());
		} else {
			appointment.setAppointmentStartDate(dataElement.getValueAsDateTime(COLUMN_STARTDATE));
			appointment.setAppointmentEndDate(dataElement.getValueAsDateTime(COLUMN_ENDDATE));
		}
	}

    @Override
    public Appointment fetchAppointment(PatientIdentifier patientIdentifier, String appointmentId) {
		DataElementDataStorage storage = DataElementDataStorage.getInstance();;
		DataElement dataElement = storage.getData(patientIdentifier.getUniqueId(), domain, appointmentId, COLUMN_UNIQUEID);
		return  createAppointment(dataElement, true);
    }

    private Appointments applyFilter(Appointments unfilteredAppointments, DateFilter dateFilter, String clinicId, String providerId) {
		Appointments filteredByDate = DateFilterer.filterByDate(unfilteredAppointments, dateFilter);
		Appointments filtered = new Appointments();
		for (Appointment potentialMatch : filteredByDate) {
			boolean match = true;
			match = match && meetsClinicFilter(potentialMatch, clinicId);
			match = match && meetsProviderFilter(potentialMatch, providerId);
			if (match) {
				filtered.add(potentialMatch);
			}
		}
		return filtered;
	}

	private boolean meetsClinicFilter(Appointment appointment, String clinicId) {
		if (NullChecker.isNullish(clinicId)) {
			return true;
		} else {
			return NullSafeStringComparer.areEqual(appointment.getClinicId(), clinicId);
		}
	}

	private boolean meetsProviderFilter(Appointment appointment, String providerId) {
		if (NullChecker.isNullish(providerId)) {
			return true;
		} else {
			return NullSafeStringComparer.areEqual(appointment.getProviderId(), providerId);
		}
	}
}
