package com.agilex.healthcare.mobilehealthplatform.datalayer.appointment;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Response.Status;

import com.agilex.healthcare.mobilehealthplatform.MockDataStorage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.facility.FacilityInternalDataLayer;
import com.agilex.healthcare.mobilehealthplatform.datalayer.facility.FacilityInternalDataLayerExcel;
import com.agilex.healthcare.mobilehealthplatform.datalayer.patient.PatientInternalDataLayer;
import com.agilex.healthcare.mobilehealthplatform.datalayer.patient.PatientInternalDataLayerExcel;
import com.agilex.healthcare.mobilehealthplatform.datalayer.xls.DataElement;
import com.agilex.healthcare.mobilehealthplatform.datalayer.xls.DataElementDataStorage;
import com.agilex.healthcare.mobilehealthplatform.datalayer.xls.DataElements;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequest;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequestDetailCode;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequestMessage;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequestMessages;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequests;
import com.agilex.healthcare.mobilehealthplatform.domain.DetailCode;
import com.agilex.healthcare.mobilehealthplatform.domain.Facilities;
import com.agilex.healthcare.mobilehealthplatform.domain.Facility;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterer;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.NullSafeStringComparer;

public class AppointmentRequestDataLayerExcel implements AppointmentRequestDataLayer {
    private static final String DEFAULT_PATIENT_ID = "D123401";
    private static final String DOMAIN = "AppointmentRequests";
    private static final String COLUMN_UNIQUEID = "UniqueId";
    private static final String COLUMN_PATIENTID = "PatientId";
    private static final String COLUMN_CREATEDDATE = "CreatedDate";
    private static final String COLUMN_LAST_UPDATED_DATE = "LastUpdatedDate";
    private static final String COLUMN_DELETEDDATE = "DeletedDate";
    private static final String COLUMN_ACTIVEFLAG = "ActiveFlag";
    private static final String COLUMN_APPOINTMENTDATE = "AppointmentDate";
    private static final String COLUMN_APPOINTMENTTIME = "AppointmentTime";
    private static final String COLUMN_OPTIONDATE1 = "OptionDate1";
    private static final String COLUMN_OPTIONTIME1 = "OptionTime1";
    private static final String COLUMN_OPTIONDATE2 = "OptionDate2";
    private static final String COLUMN_OPTIONTIME2 = "OptionTime2";
    private static final String COLUMN_OPTIONDATE3 = "OptionDate3";
    private static final String COLUMN_OPTIONTIME3 = "OptionTime3";
    private static final String COLUMN_STATUS = "Status";
    private static final String COLUMN_APPOINTMENTTYPE = "AppointmentType";
    private static final String COLUMN_FACILITYCODE = "FacilityCode";
    private static final String COLUMN_EMAIL = "Email";
    private static final String COLUMN_PHONENUMBER = "PhoneNumber";
    private static final String COLUMN_PREFERRED_CONTACT_TIME = "PreferredContactTime";
    private static final String COLUMN_PURPOSEOFVISIT = "PurposeOfVisit";
    private static final String COLUMN_OTHERPURPOSEOFVISIT = "OtherPurposeOfVisit";
    private static final String COLUMN_PROVIDERID = "ProviderId";
    private static final String COLUMN_TEXTMESSAGINGALLOWED = "TextMessagingAllowed";
    private static final String COLUMN_TEXTMESSAGINGPHONENUMBER = "TextMessagingPhoneNumber";
    private static final String COLUMN_SECONDREQUEST = "SecondRequest";
    private static final String COLUMN_ARDETAILCODE = "ARDetailCode";
    private static final String COLUMN_DETAILCODE = "DetailCode";
    private static final String COLUMN_LASTUPDATEDUSERID = "LastUpdatedUserId";
    private static final String COLUMN_PROVIDERNAME = "ProviderName";
    private static final String COLUMN_PROVIDEROPTION = "ProviderOption";
    private static final String COLUMN_VISITTYPE = "VisitType";

    private static Set<DetailCode> detailCodes;
    private static Facilities facilities;

    private Map<String, List<AppointmentRequestMessage>> mockAppointmentRequestMessagesStorage = new HashMap<String,List<AppointmentRequestMessage>>();
    private static Integer mockAptRequestMessageUniqueId = 0;
    static{
        DetailCodeDataLayer detailCodeDataLayer = new DetailCodeDataLayerExcel();
        detailCodes = detailCodeDataLayer.fetchDetailCodes();

        FacilityInternalDataLayer facilityInternalDataLayer = new FacilityInternalDataLayerExcel();
        facilities = facilityInternalDataLayer.fetchFacilities();
    }

    private static MockDataStorage<AppointmentRequests, AppointmentRequest> data = new MockDataStorage<AppointmentRequests, AppointmentRequest>() {
        @Override
        protected AppointmentRequests createDataCollection() {
            return new AppointmentRequests();
        }
    };

    private void init(String patientId) {
        if (!data.dataExists(patientId)) {
            DataElementDataStorage storage = DataElementDataStorage.getInstance();
            DataElements dataElements = storage.getData(patientId, DOMAIN);
            data.setDataElements(createAppointmentRequests(dataElements));
        }
    }

    @Override
    public AppointmentRequests fetchAppointmentRequestsByPatient(String patientId, DateFilter filter) {
        init(patientId);
        AppointmentRequests appointmentRequests = data.getDataForPatient(patientId);

        Iterator<AppointmentRequest> iterator = appointmentRequests.iterator();
        while (iterator.hasNext()) {
            AppointmentRequest appointmentRequest = iterator.next();
            if (appointmentRequest.isActiveFlag() == false) {
                iterator.remove();
            } else {
                Collections.sort(appointmentRequest.getAppointmentRequestDetailCode());
            }
        }

        return DateFilterer.filterByDate(appointmentRequests, filter);
    }

    @Override
    public AppointmentRequest fetchAppointmentRequest(String patientId, String appointmentRequestId) {
        init(patientId);
        AppointmentRequests allAppointmentRequests = fetchAppointmentRequestsByPatient(patientId, DateFilterFactory.createEmptyFilter());
        AppointmentRequest match = null;

        for (AppointmentRequest appointmentRequest : allAppointmentRequests) {
            if (NullSafeStringComparer.areEqual(appointmentRequestId, appointmentRequest.getUniqueId())) {
                if (appointmentRequest.isActiveFlag()) {
                    match = appointmentRequest;
                    Collections.sort(match.getAppointmentRequestDetailCode());
                    break;
                }
            }
        }
        if (match == null) {
            throw new WebApplicationException(Status.NOT_FOUND);
        }
        return match;
    }

    @Override
    public AppointmentRequestMessages fetchAppointmentRequestMessages(String patientId, String appointRequestId) {
        AppointmentRequestMessages appointmentRequestMessages = new AppointmentRequestMessages();

        if(mockAppointmentRequestMessagesStorage.get(appointRequestId)!=null){
            List<AppointmentRequestMessage> aptRequestMessages = mockAppointmentRequestMessagesStorage.get(appointRequestId);
            for (AppointmentRequestMessage aptRequestMessage : aptRequestMessages) {
            	if (aptRequestMessage.getPatientId().equals(patientId)) {
            		appointmentRequestMessages.add(aptRequestMessage);
            	}
            }
        }

        return appointmentRequestMessages;
    }

    @Override
    public AppointmentRequestMessage saveAppointmentRequestMessage(String patientId, AppointmentRequestMessage appointmentRequestMessage, boolean isProvider) {
    	
    	if(mockAppointmentRequestMessagesStorage.get(appointmentRequestMessage.getAppointmentRequestId()) == null){
            List<AppointmentRequestMessage> aptRequestMessages = new ArrayList<AppointmentRequestMessage>();
            mockAppointmentRequestMessagesStorage.put(appointmentRequestMessage.getAppointmentRequestId(), aptRequestMessages);
        }

    	appointmentRequestMessage.setPatientId(patientId);
        appointmentRequestMessage.setUniqueId(mockAptRequestMessageUniqueId.toString()); //message MUST have uniqueId otherwise crashes later. Mocking it here.
        mockAptRequestMessageUniqueId++;

        mockAppointmentRequestMessagesStorage.get(appointmentRequestMessage.getAppointmentRequestId()).add(appointmentRequestMessage);

        return appointmentRequestMessage;
    }

    private AppointmentRequests createAppointmentRequests(DataElements dataElements) {
        AppointmentRequests appointmentRequests = new AppointmentRequests();
        for (DataElement dataElement : dataElements) {

            if (isDataElementAppointmentRequest(dataElement)) {
                AppointmentRequest appointmentRequest = createAppointmentRequest(dataElement);
                appointmentRequests.add(appointmentRequest);
            } else if(isDataElementAppointmentRequestDetailCode(dataElement)) {
                AppointmentRequest appointmentRequest = getAppointmentRequest(appointmentRequests, dataElement.getValue(COLUMN_UNIQUEID));
                createAppointmentRequestDetailCode(appointmentRequest, dataElement);
            } else{
                // Best Time to Call
                AppointmentRequest appointmentRequest = getAppointmentRequest(appointmentRequests, dataElement.getValue(COLUMN_UNIQUEID));
                appointmentRequest.getBestTimetoCall().add(dataElement.getValue(COLUMN_PREFERRED_CONTACT_TIME));
            }
        }

        return appointmentRequests;
    }

    private void createAppointmentRequestDetailCode(AppointmentRequest appointmentRequest, DataElement dataElement) {
        AppointmentRequestDetailCode arDetailCode = new AppointmentRequestDetailCode();
        arDetailCode.setCreatedDate(dataElement.getValueAsDateTime(COLUMN_CREATEDDATE));
        arDetailCode.setUserId(dataElement.getValue(COLUMN_LASTUPDATEDUSERID));
        arDetailCode.setAppointmentRequestDetailCodeId(dataElement.getValue(COLUMN_ARDETAILCODE));
        arDetailCode.setDetailCode(getDetailCode(dataElement.getValue(COLUMN_DETAILCODE)));
        addARDetailCodeToAppointmentRequest(appointmentRequest, arDetailCode);
    }

    private void addARDetailCodeToAppointmentRequest(AppointmentRequest appointmentRequest, AppointmentRequestDetailCode arDetailCode) {

        List<AppointmentRequestDetailCode> arDetailCodes = appointmentRequest.getAppointmentRequestDetailCode();

        if(arDetailCodes == null){
            arDetailCodes = new LinkedList<AppointmentRequestDetailCode>();
        }

        arDetailCodes.add(arDetailCode);
        appointmentRequest.setAppointmentRequestDetailCode(arDetailCodes);
    }

    private DetailCode getDetailCode(String code) {
        for (DetailCode detailCode : detailCodes) {
            if(detailCode.getCode().equalsIgnoreCase(code)){
                return detailCode;
            }
        }

        return null;
    }

    private Facility getFacility(String code) {
        Facility result = null;
        for (Facility facility : facilities) {
            if(facility.getFacilityCode().equalsIgnoreCase(code)){
                result = facility;
                break;
            }
        }

        return result;
    }

    private boolean isDataElementAppointmentRequestDetailCode(DataElement dataElement) {
        return dataElement.getValue(COLUMN_ARDETAILCODE).contentEquals("") == false;
    }

    private AppointmentRequest getAppointmentRequest(AppointmentRequests appointmentRequests, String id) {
        AppointmentRequest match = null;
        for (AppointmentRequest potentialMatch : appointmentRequests) {
            if (potentialMatch.getUniqueId().contentEquals(id)) {
                match = potentialMatch;
                break;
            }
        }
        return match;
    }

    private boolean isDataElementAppointmentRequest(DataElement dataElement) {
        return dataElement.getValue(COLUMN_APPOINTMENTTYPE).contentEquals("") == false;
    }

    private AppointmentRequest createAppointmentRequest(DataElement dataElement) {
        AppointmentRequest a = new AppointmentRequest();
        a.setUniqueId(dataElement.getValue(COLUMN_UNIQUEID));
        a.setPatientId(dataElement.getValue(COLUMN_PATIENTID));
        a.setCreatedDate(dataElement.getValueAsDateTime(COLUMN_CREATEDDATE));
        a.setLastUpdatedDate(dataElement.getValueAsDateTime(COLUMN_LAST_UPDATED_DATE));
        a.setDeletedDate(dataElement.getValueAsDateTime(COLUMN_DELETEDDATE));
        a.setActiveFlag(dataElement.getValueAsBoolean(COLUMN_ACTIVEFLAG));
        a.setStatus(dataElement.getValue(COLUMN_STATUS));
        a.setAppointmentType(dataElement.getValue(COLUMN_APPOINTMENTTYPE));
        a.setEmail(dataElement.getValue(COLUMN_EMAIL));
        a.setPhoneNumber(dataElement.getValue(COLUMN_PHONENUMBER));
        a.setAppointmentDate(dataElement.getValue(COLUMN_APPOINTMENTDATE));
        a.setAppointmentTime(dataElement.getValue(COLUMN_APPOINTMENTTIME));
        a.setOptionDate1(dataElement.getValue(COLUMN_OPTIONDATE1));
        a.setOptionTime1(dataElement.getValue(COLUMN_OPTIONTIME1));
        a.setOptionDate2(dataElement.getValue(COLUMN_OPTIONDATE2));
        a.setOptionTime2(dataElement.getValue(COLUMN_OPTIONTIME2));
        a.setOptionDate3(dataElement.getValue(COLUMN_OPTIONDATE3));
        a.setOptionTime3(dataElement.getValue(COLUMN_OPTIONTIME3));
        a.setPurposeOfVisit(dataElement.getValue(COLUMN_PURPOSEOFVISIT));
        a.setOtherPurposeOfVisit(dataElement.getValue(COLUMN_OTHERPURPOSEOFVISIT));
        a.setProviderId(dataElement.getValue(COLUMN_PROVIDERID));
        a.setProviderName(dataElement.getValue(COLUMN_PROVIDERNAME));
        a.setProviderOption(dataElement.getValue(COLUMN_PROVIDEROPTION));
        a.setVisitType(dataElement.getValue(COLUMN_VISITTYPE));
        a.setTextMessagingAllowed(dataElement.getValueAsBoolean(COLUMN_TEXTMESSAGINGALLOWED));
        a.setTextMessagingPhoneNumber(dataElement.getValue(COLUMN_TEXTMESSAGINGPHONENUMBER));

        Facility facility = getFacility(dataElement.getValue(COLUMN_FACILITYCODE));
        a.setFacility(facility);

        PatientInternalDataLayer patientInternalDataLayer = new PatientInternalDataLayerExcel();
        Patient patient = patientInternalDataLayer.fetchPatientById(dataElement.getValue(COLUMN_PATIENTID));

        a.setPatient(patient);
        a.setSecondRequest(dataElement.getValueAsBoolean(COLUMN_SECONDREQUEST));

        return a;
    }

    @Override
    public AppointmentRequest saveAppointmentRequest(AppointmentRequest appointmentRequest) {

        List<AppointmentRequestDetailCode> appointmentRequestDetailCode = appointmentRequest.getAppointmentRequestDetailCode();

        for (AppointmentRequestDetailCode arDetailCode : appointmentRequestDetailCode) {
            DetailCode detailCode = getDetailCode(arDetailCode.getDetailCode().getCode());
            arDetailCode.setDetailCode(detailCode);
        }

        String facilityCode = appointmentRequest.getFacility().getFacilityCode();
        Facility facility = getFacility(facilityCode);
        appointmentRequest.setFacility(facility);

        FacilityInternalDataLayerExcel facilityInternalDao = new FacilityInternalDataLayerExcel();
        facilityInternalDao.saveFacility(facility);

        PatientInternalDataLayer patientInternalDataLayer = new PatientInternalDataLayerExcel();
        patientInternalDataLayer.savePatient(appointmentRequest.getPatient());

        // Check for parent request and update it if it exists
        if(appointmentRequest.getParentRequest() != null){
            setAppointmentRequest(appointmentRequest.getParentRequest());
        }

        appointmentRequest.setParentRequest(null);

        setAppointmentRequest(appointmentRequest);

        return appointmentRequest;
    }

    private static synchronized void setAppointmentRequest(AppointmentRequest appointmentRequest) {
        data.setDataElement(appointmentRequest);
    }

    @Override
    public void deleteAppointmentRequest(AppointmentRequest appointmentRequest) {
        appointmentRequest.setActiveFlag(false);

        if (NullChecker.isNullish(appointmentRequest.getDeletedDate())) {
            appointmentRequest.setDeletedDate(new Date());
        }

        saveAppointmentRequest(appointmentRequest);
    }

    @Override
    public AppointmentRequests fetchAppointmentRequests(AppointmentRequestFilter filter) {
        init(DEFAULT_PATIENT_ID);

        AppointmentRequests allAppointmentRequests = data.getAllRows();

        AppointmentRequests filteredAppointmentRequests = applyAppointmentRequestFilter(allAppointmentRequests, filter);

        return filteredAppointmentRequests;
    }

    private AppointmentRequests applyAppointmentRequestFilter(AppointmentRequests allAppointmentRequests, AppointmentRequestFilter filter) {
        AppointmentRequests filteredAppointmentRequests = new AppointmentRequests();
        boolean checkFacilityName = filter == null ? false : NullChecker.isNotNullish(filter.getFacilityName());
        boolean checkParentSiteCode = filter == null ? false : NullChecker.isNotNullish(filter.getParentSiteCode());

        if(filter == null || (!checkFacilityName && !checkParentSiteCode)){
            return allAppointmentRequests;
        }else{
            for (AppointmentRequest appointmentRequest : allAppointmentRequests) {
                boolean match = false;

                if (checkFacilityName) {
                    if(filter.getFacilityName().equalsIgnoreCase(appointmentRequest.getFacility().getName())){
                        match = true;
                    }
                }
                if (!match && checkParentSiteCode) {
                    if(filter.getParentSiteCode().equalsIgnoreCase(appointmentRequest.getFacility().getParentSiteCode())){
                        match = true;
                    }
                }

                if (match) {
                    filteredAppointmentRequests.add(appointmentRequest);
                }
            }
        }

        filteredAppointmentRequests = DateFilterer.filterByDate(filteredAppointmentRequests, filter);

        return filteredAppointmentRequests;
    }

	@Override
	public AppointmentRequest updateMessageFlag(String patientId, String appointmentRequestId,
			boolean isProvider) {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public AppointmentRequest markAppointmentRequestIsSeenByProvider(
			String patientId,
			String appointmentRequestId) {
		// TODO Auto-generated method stub
		return null;
	}

}
