package com.agilex.healthcare.mobilehealthplatform;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import com.agilex.healthcare.mobilehealthplatform.domain.PatientData;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientDataCollection;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.NullSafeStringComparer;

public abstract class MockDataStorage<Ts extends PatientDataCollection<T>, T extends PatientData> {
	private Map<String, Ts> data = new HashMap<String, Ts>();
	
	public Ts getData() {
		Ts result = createDataCollection();
		for (String key : data.keySet()) {
			result.addAll(data.get(key));
		}
		return result;
	}
	
	public Ts getDataForPatient(String patientId) {
		Ts patientData = data.get(patientId);
		if (patientData == null)
			patientData = createDataCollection();
		return patientData;
	}

	public boolean dataExists(String patientId) {
		return data.containsKey(patientId);
	}

	public synchronized void setDataElement(T dataElement) {
		if (dataElement == null)
			throw new IllegalArgumentException();
		if (NullChecker.isNullish(dataElement.getPatientId()))
			throw new IllegalArgumentException();

		Ts dataCollection = data.get(dataElement.getPatientId());
		if (dataCollection == null) {
			dataCollection = createDataCollection();
			data.put(dataElement.getPatientId(), dataCollection);
		}

		if (NullChecker.isNullish(dataElement.getUniqueId())) {
			// insert
			dataElement.setUniqueId(UUID.randomUUID().toString());
		} else {
			T tToRemoveForUpdate = null;
			for (T candidate : dataCollection) {
				if (NullSafeStringComparer.areEqual(candidate.getUniqueId(), dataElement.getUniqueId())) {
					tToRemoveForUpdate = candidate;
				}
			}

			if (tToRemoveForUpdate != null) {
				dataCollection.remove(tToRemoveForUpdate);
			}
		}
		dataCollection.add(dataElement);
	}

	public synchronized void setDataElements(Ts dataElements) {
		for (T dataElement : dataElements) {
			setDataElement(dataElement);
		}
	}

	protected abstract Ts createDataCollection();
	
	public Ts getAllRows(){
		Ts allRows = createDataCollection();
		Collection<Ts> TsCollection = data.values();

		for (Ts ts : TsCollection) {
			allRows.addAll(ts);
		}
		
		return allRows;
	}
}
