package com.agilex.healthcare.vamf.cdw.vitals;

import com.agilex.healthcare.mobilehealthplatform.datalayer.vital.VitalDataLayer;
import com.agilex.healthcare.mobilehealthplatform.datalayer.vital.VitalDataLayerFilterer;
import com.agilex.healthcare.mobilehealthplatform.datalayer.vital.VitalEntryFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalEntries;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalEntry;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.PatientIdentityHelper;
import org.springframework.stereotype.Repository;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.TypedQuery;
import java.util.Date;
import java.util.List;

@Repository
public class CdwVitalsDataLayer implements VitalDataLayer {

    @PersistenceContext(unitName="cdwPersistenceUnit")
    protected EntityManager entityManager;

    private static final String WHERE_PATIENT_ID_DATE_RANGE = "FROM CdwVitalEntryPo where patientIcn = :patientIcn and dateTaken > :startDate and dateTaken < :endDate";


    @Override
    public VitalEntries fetchVitalEntries(PatientIdentifier patientIdentifier, VitalEntryFilter filter) {
        // CDW does not use the checksum for data queries
        PatientIdentifier truncatedIcn = truncateICN(patientIdentifier);

        final TypedQuery<CdwVitalEntryPo> query = this.entityManager.createQuery(WHERE_PATIENT_ID_DATE_RANGE, CdwVitalEntryPo.class);
        query.setParameter("patientIcn", truncatedIcn.getUniqueId());
        query.setParameter("startDate", startDateOrDefault(filter));
        query.setParameter("endDate", endDateOrDefault(filter));
        List<CdwVitalEntryPo> cdwVitalEntries = query.getResultList();
        VitalEntries vitalEntries = createVitalEntries(cdwVitalEntries);

        resetPatientIdsForVitalEntries(patientIdentifier, vitalEntries);

        return filter(vitalEntries, filter);
    }

    @Override
    public VitalEntry fetchVitalEntry(PatientIdentifier patientIdentifier, String vitalId) {
        return null;
    }

    @Override
    public VitalEntry saveVitalEntry(VitalEntry vitalEntry) {
        throw new UnsupportedOperationException();
    }

    @Override
    public void deleteVitalEntry(VitalEntry vitalEntry) {
        throw new UnsupportedOperationException();
    }

    @Override
    public VitalEntries getVitalEntries(String patientId, VitalEntryFilter filter) {
        throw new UnsupportedOperationException();
    }

    @Override
    public VitalEntry getVitalEntry(String patientId, String vitalId) {
        throw new UnsupportedOperationException();
    }

    PatientIdentifier truncateICN(PatientIdentifier patientIdentifier) {
        PatientIdentityHelper helper = new PatientIdentityHelper();
        return helper.truncateICN(patientIdentifier);
    }

    void resetPatientIdsForVitalEntries(PatientIdentifier patientIdentifier, VitalEntries vitalEntries) {
        for(VitalEntry vitalEntry : vitalEntries) {
            vitalEntry.setPatientIdentifier(patientIdentifier);
        }
    }

    VitalEntries filter(VitalEntries vitalEntries, VitalEntryFilter filter) {
        return VitalDataLayerFilterer.applyFilter(vitalEntries, filter);
    }

    VitalEntries createVitalEntries(List<CdwVitalEntryPo> cdwVitalEntires) {
        VitalEntries vitalEntries = new VitalEntries();
        if(NullChecker.isNotNullish(cdwVitalEntires)) {
            for(CdwVitalEntryPo po : cdwVitalEntires) {
                vitalEntries.add(po.createVitalEntry());
            }
        }
        return vitalEntries;
    }

    Date startDateOrDefault(VitalEntryFilter filter) {
        // If no start date is provided, filter to 2 years of data
        if(filter == null || filter.getStartDate() == null)
            return DateHelper.minusMonths(new Date(), 24);
        return filter.getStartDate();
    }

    Date endDateOrDefault(VitalEntryFilter filter) {
        // if no end date is provided, assume vitals until today's date
        if(filter == null || filter.getEndDate() == null)
            return new Date();
        return filter.getEndDate();
    }
}
