package com.agilex.healthcare.vamf.cdw.problems;

import java.util.Date;
import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.TypedQuery;

import org.springframework.stereotype.Repository;

import com.agilex.healthcare.mobilehealthplatform.datalayer.problem.ProblemDataLayer;
import com.agilex.healthcare.mobilehealthplatform.datalayer.vital.VitalEntryFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.Problems;
import com.agilex.healthcare.mobilehealthplatform.domain.code.SourceSystemCode;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.PatientIdentityHelper;

@Repository
public class CdwProblemDataLayer implements ProblemDataLayer {

	@PersistenceContext(unitName="cdwPersistenceUnit")
    protected EntityManager entityManager;
	
	private static final String PROBLEM_QUERY_BY_DATE = "FROM ProblemPo WHERE patientIcn = :patientIcn and onsetDate >= :startDate and onsetDate <= :endDate";
	private static final String PROBLEM_QUERY = "FROM ProblemPo WHERE patientIcn = :patientIcn";
	
	@Override
	public Problems fetchProblems(PatientIdentifier patientIdentifier, DateFilter dateFilter) {
		// CDW does not use the checksum for data queries - stripping it out everything after the 'V'
		// e.g. 1011888555V123456 -> 1011888555
		PatientIdentifier truncatedIcn = truncateICN(patientIdentifier);
		TypedQuery<ProblemPo> query = this.entityManager.createQuery(PROBLEM_QUERY, ProblemPo.class);
		query.setParameter("patientIcn", truncatedIcn.getUniqueId());
		List<ProblemPo> problemResults = query.getResultList();
		
		return toProblems(problemResults, patientIdentifier);
	}
	
	Problems toProblems(List<ProblemPo> problemPos, PatientIdentifier patientIdentifier) {
		Problems problems = new Problems();
		if(NullChecker.isNullish(problemPos))
			return problems;

		for(ProblemPo po : problemPos) {
			problems.add(po.createProblem());			
		}
		problems.updateSystemId(SourceSystemCode.CDW);
		return problems;
	}
	
	PatientIdentifier truncateICN(PatientIdentifier patientIdentifier) {
        PatientIdentityHelper helper = new PatientIdentityHelper();
        return helper.truncateICN(patientIdentifier);
    }
	
	Date startDateOrDefault(DateFilter dateFilter) {
        // If no start date is provided, filter to 4 years of data
        if(dateFilter == null || dateFilter.getStartDate() == null)
            return DateHelper.minusMonths(new Date(), 48);
        return dateFilter.getStartDate();
    }

    Date endDateOrDefault(DateFilter dateFilter) {
        // if no end date is provided, assume vitals until today's date
        if(dateFilter == null || dateFilter.getEndDate() == null)
            return new Date();
        return dateFilter.getEndDate();
    }

}
