package com.agilex.healthcare.vamf.cdw.chemlabs;

import com.agilex.healthcare.mobilehealthplatform.datalayer.lab.ChemistryLabDataLayer;
import com.agilex.healthcare.mobilehealthplatform.domain.LabResult;
import com.agilex.healthcare.mobilehealthplatform.domain.LabResults;
import com.agilex.healthcare.mobilehealthplatform.domain.LabTestGroups;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.PatientIdentityHelper;
import org.springframework.stereotype.Repository;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.TypedQuery;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Repository
public class CdwChemistryLabDataLayer implements ChemistryLabDataLayer {

    @PersistenceContext(unitName="cdwPersistenceUnit")
    protected EntityManager entityManager;

    private static final String WHERE_PATIENT_ID_DATE_RANGE = "FROM CdwChemLabResultPo where patientIcn = :patientIcn and completionDate >= :startDate and completionDate <= :endDate";
    private static final String WHERE_PATIENT_ID_AND_TEST_ID = "FROM CdwChemLabResultPo where patientICN = :patientIcn and loinc = :loinc";

    @Override
    public LabResults getLabResults(PatientIdentifier patientIdentifier, DateFilter dateFilter) {

        // CDW does not use the checksum for data queries
        PatientIdentifier truncatedIcn = truncateICN(patientIdentifier);

        TypedQuery<CdwChemLabResultPo> query = this.entityManager.createQuery(WHERE_PATIENT_ID_DATE_RANGE, CdwChemLabResultPo.class);
        query.setParameter("patientIcn", truncatedIcn.getUniqueId());
        query.setParameter("startDate", startDateOrDefault(dateFilter));
        query.setParameter("endDate", endDateOrDefault(dateFilter));
        List<CdwChemLabResultPo> cdwChemLabs = query.getResultList();
        LabResults labResults = createLabResults(cdwChemLabs);

        return labResults;
    }

    @Override
    public LabResults getLabResultsByTestId(PatientIdentifier patientIdentifier, String testId) {
        // CDW does not use the checksum for data queries
        PatientIdentifier truncatedIcn = truncateICN(patientIdentifier);

        TypedQuery<CdwChemLabResultPo> query = this.entityManager.createQuery(WHERE_PATIENT_ID_AND_TEST_ID, CdwChemLabResultPo.class);
        query.setParameter("patientIcn", truncatedIcn.getUniqueId());
        query.setParameter("loinc", testId);
        List<CdwChemLabResultPo> cdwChemLabs = query.getResultList();

        LabResults labResults = createLabResults(cdwChemLabs);
        resetPatientIdentifiers(patientIdentifier, labResults);

        return labResults;
    }

    @Override
    public LabResults getLabResultsLite(String patientId, DateFilter dateFilter) {
        return new LabResults();
    }

    @Override
    public Map<String, LabTestGroups> lookupLabGroupsByTestIds(List<String> labTestIds) {
        return new HashMap<String, LabTestGroups>();
    }

    @Override
    public LabTestGroups lookupLabGroupsByTestId(String labTestId) {
        return new LabTestGroups();
    }

    LabResults createLabResults(List<CdwChemLabResultPo> cdwChemLabs) {
        LabResults labResults = new LabResults();

        if(NullChecker.isNotNullish(cdwChemLabs)) {
            for(CdwChemLabResultPo po : cdwChemLabs) {
                labResults.add(po.createLabResult());
            }
        }

        return labResults;
    }

    void resetPatientIdentifiers(PatientIdentifier patientidentifier, LabResults labResults) {
        for(LabResult labResult : labResults) {
            labResult.setPatientIdentifier(patientidentifier);
        }
    }

    PatientIdentifier truncateICN(PatientIdentifier patientIdentifier) {
        PatientIdentityHelper helper = new PatientIdentityHelper();
        return helper.truncateICN(patientIdentifier);
    }

    Date startDateOrDefault(DateFilter filter) {
        // If no start date is provided, filter to 12 years of data
        if(filter == null || filter.getStartDate() == null)
            return DateHelper.minusMonths(new Date(), 144);
        return filter.getStartDate();
    }

    Date endDateOrDefault(DateFilter filter) {
        // if no end date is provided, assume vitals until today's date
        if(filter == null || filter.getEndDate() == null)
            return new Date();
        return filter.getEndDate();
    }
}
