package com.agilex.healthcare.vamf.cdw.careteam;

import com.agilex.healthcare.mobilehealthplatform.datalayer.provider.ProviderDataLayer;
import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.PatientIdentityHelper;
import org.springframework.stereotype.Repository;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.TypedQuery;
import java.util.Date;
import java.util.List;

@Repository
public class CdwPatientCareTeamDataLayer implements ProviderDataLayer {

    @PersistenceContext(unitName="cdwPersistenceUnit")
    protected EntityManager entityManager;

    private static final String WHERE_PATIENT_ID_CARE_TEAM = "FROM PatientCareTeamPO where patientIcn = :patientIcn and sta3n = :siteCode";
    private static final String WHERE_PATIENT_ID_MH_VISITS = "FROM MentalHealthVisitPO where patientICN = :patientIcn and sta3n = :siteCode and visitDateTime >= :startDate and visitDateTime <= :endDate";



    @Override
    public PatientProviders fetchPrimaryCareProviders(PatientIdentifier patientIdentifier, String siteCode) {
        PatientIdentifier truncatedIcn = truncateICN(patientIdentifier);

        TypedQuery<PatientCareTeamPO> query = this.entityManager.createQuery(WHERE_PATIENT_ID_CARE_TEAM, PatientCareTeamPO.class);
        query.setParameter("patientIcn", truncatedIcn.getUniqueId());
        query.setParameter("siteCode", siteCode);

        List<PatientCareTeamPO> patientCareTeam = query.getResultList();

        PatientProviders patientProviders = createPatientProvidersFromCareTeam(patientCareTeam);
        resetPatientIdentifiers(patientIdentifier, patientProviders);

        return patientProviders;
    }

    public PatientProviders fetchMentalHealthProviders(PatientIdentifier patientIdentifier, String siteCode) {

        return fetchMentalHealthProviders(patientIdentifier, null, siteCode);
    }

    @Override
    public PatientProviders fetchMentalHealthProviders(PatientIdentifier patientIdentifier, DateFilter  filter, String siteCode) {
        PatientIdentifier truncatedIcn = truncateICN(patientIdentifier);

        TypedQuery<MentalHealthVisitPO> query = this.entityManager.createQuery(WHERE_PATIENT_ID_MH_VISITS, MentalHealthVisitPO.class);
        query.setParameter("patientIcn", truncatedIcn.getUniqueId());
        query.setParameter("siteCode", siteCode);
        query.setParameter("startDate", startDateOrDefault(filter));
        query.setParameter("endDate", endDateOrDefault(filter));

        List<MentalHealthVisitPO> visit = query.getResultList();

        PatientProviders patientProviders = createPatientProvidersFromMHVisits(visit);
        resetPatientIdentifiers(patientIdentifier, patientProviders);

        return patientProviders;
    }

    @Deprecated
    @Override
    public PatientProviders fetchPrimaryCareProviders(PatientIdentifier patientIdentifier) {
        return null;
    }


    PatientProviders createPatientProvidersFromCareTeam(List<PatientCareTeamPO> careTeam) {
        PatientProviders patientProviders = new PatientProviders();

        if(NullChecker.isNotNullish(careTeam)) {
            for(PatientCareTeamPO teamMember : careTeam) {
                PatientProvider     provider;

                provider = new PatientProvider();
                provider.setProviderName(teamMember.getStaffName());
                provider.setPersonClass(teamMember.getPrimaryStandardPosition());
                provider.setVisitDate(teamMember.getStartDate());
                patientProviders.add(provider);
            }
        }

        return patientProviders;
    }

    PatientProviders createPatientProvidersFromMHVisits(List<MentalHealthVisitPO> mhVisits) {
        PatientProviders patientProviders = new PatientProviders();

        if(NullChecker.isNotNullish(mhVisits)) {
            for(MentalHealthVisitPO visit : mhVisits) {
                PatientProvider     provider;

                provider = new PatientProvider();
                provider.setProviderName(visit.getFirstName() + " " + visit.getLastName());
                provider.setPersonClass(visit.getProviderClass());
                provider.setVisitDate(visit.getVisitDateTime());
                patientProviders.add(provider);
            }
        }

        return patientProviders;
    }

    void resetPatientIdentifiers(PatientIdentifier patientidentifier, PatientProviders providers) {
        for(PatientProvider provider : providers) {
            provider.setPatientIdentifier(patientidentifier);
        }
    }


    Date startDateOrDefault(DateFilter filter) {
        // If no start date is provided, filter to 12 years of data
        if(filter == null || filter.getStartDate() == null)
            return DateHelper.minusMonths(new Date(), 24);
        return filter.getStartDate();
    }

    Date endDateOrDefault(DateFilter filter) {
        // if no end date is provided, assume vitals until today's date
        if(filter == null || filter.getEndDate() == null)
            return new Date();
        return filter.getEndDate();
    }

    PatientIdentifier truncateICN(PatientIdentifier patientIdentifier) {
        PatientIdentityHelper helper = new PatientIdentityHelper();
        return helper.truncateICN(patientIdentifier);
    }
}
