package com.agilex.healthcare.vamf.cdw.appointment;

import java.util.Date;
import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.TypedQuery;

import org.springframework.stereotype.Repository;

import com.agilex.healthcare.mobilehealthplatform.datalayer.appointment.AppointmentDataLayer;
import com.agilex.healthcare.mobilehealthplatform.datalayer.appointment.MentalHealthAppointmentDataLayer;
import com.agilex.healthcare.mobilehealthplatform.domain.Appointment;
import com.agilex.healthcare.mobilehealthplatform.domain.Appointments;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.code.SourceSystemCode;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.PatientIdentityHelper;


@Repository
public class CdwAppointmentDataLayer implements AppointmentDataLayer, MentalHealthAppointmentDataLayer {
	
	@PersistenceContext(unitName="cdwPersistenceUnit")
    protected EntityManager entityManager;
	
	private static final String MHAPPOINTMENTS_QUERY = "FROM MentalHealthAppointmentPo where patientIcn = :patientIcn";
	private static final String MHAPPOINTMENTS_BY_DATE_QUERY = MHAPPOINTMENTS_QUERY + " AND appointmentStartDate between :startDate and :endDate";
	
	private static final String APPOINTMENTS_QUERY = "FROM AppointmentPo where patientIcn = :patientIcn";
	private static final String APPOINTMENTS_BY_DATE_QUERY = APPOINTMENTS_QUERY + " AND appointmentStartDate between :startDate and :endDate";
	private static final String APPOINTMENT_QUERY = "FROM AppointmentPo where patientIcn = :patientIcn and id = :id";
	
	private static final int SIX_MONTHS_IN_DAYS = 180;
	
	@Override
	public Appointments getProviderAppointments(String providerId, DateFilter filter) {
		throw new UnsupportedOperationException();
	}

	@Override
	public Appointments getClinicAppointments(String clinicId, DateFilter filter) {
		throw new UnsupportedOperationException();
	}

	@Override
	public Appointments fetchAppointmentsByPatient(PatientIdentifier patientIdentifier, DateFilter filter) {
		
		List<AppointmentPo> appointmentPos = null;
		String patientICN = getPatientICN(patientIdentifier);
		
		if (filter == null || filter.getStartDate() == null){
			appointmentPos = getPatientAppointments(patientICN);
		}else{
			Date startDate = filter.getStartDate();
			Date endDate = filter.getEndDate();
			if (endDate == null){
				endDate = DateHelper.plusDays(startDate, SIX_MONTHS_IN_DAYS);
			}
			appointmentPos = getPatientAppointments(patientICN, startDate, endDate);
		}	
		return translateToAppointments(appointmentPos, patientIdentifier);
	}
	
	/**
	 * @deprecated  @shouldIncludeDetails parameter is ignored. Notes are now requested separately via ClinicalNoteDataLayer. 
	 * This overload method is no longer needed.
	 */
	@Deprecated
	@Override
	public Appointments fetchAppointmentsByPatient(PatientIdentifier patientIdentifier, DateFilter filter, boolean shouldIncludeDetails) {
		return fetchAppointmentsByPatient(patientIdentifier, filter);
	}
	
	@Override
	public Appointment fetchAppointment(PatientIdentifier patientIdentifier, String appointmentId) {
		AppointmentPo appointmentPo = getPatientAppointment(getPatientICN(patientIdentifier), appointmentId);
		return appointmentPo.translateToAppointment();
	}

	@Override
	public Appointments fetchMentalHealthAppointments(PatientIdentifier patientIdentifier, DateFilter filter) {
		List<MentalHealthAppointmentPo> appointmentPos = null;
		String patientICN = getPatientICN(patientIdentifier);
		
		if (filter == null || filter.getStartDate() == null){
			appointmentPos = getPatientMHAppointments(patientICN);
		}else{
			Date startDate = filter.getStartDate();
			Date endDate = filter.getEndDate();
			if (endDate == null){
				endDate = DateHelper.plusDays(startDate, SIX_MONTHS_IN_DAYS);
			}
			appointmentPos = getPatientMHAppointments(patientICN, startDate, endDate);
		}	
		return translateToAppointments(appointmentPos, patientIdentifier);
	}	 
	
	private List<MentalHealthAppointmentPo> getPatientMHAppointments(String patientICN){
		
		final TypedQuery<MentalHealthAppointmentPo> query = this.entityManager.createQuery(MHAPPOINTMENTS_QUERY, MentalHealthAppointmentPo.class);
		query.setParameter("patientIcn", patientICN);
		List<MentalHealthAppointmentPo> appointmentPos = query.getResultList();
		return appointmentPos;
		
	}
	
	private List<MentalHealthAppointmentPo> getPatientMHAppointments(String patientICN, Date startDate, Date endDate){
		
		final TypedQuery<MentalHealthAppointmentPo> query = this.entityManager.createQuery(MHAPPOINTMENTS_BY_DATE_QUERY, MentalHealthAppointmentPo.class);
		query.setParameter("patientIcn", patientICN);
		query.setParameter("startDate", startDate);
		query.setParameter("endDate", endDate);
		List<MentalHealthAppointmentPo> appointmentPos = query.getResultList();
		return appointmentPos;
		
	}

	private List<AppointmentPo> getPatientAppointments(String patientICN){
		
		final TypedQuery<AppointmentPo> query = this.entityManager.createQuery(APPOINTMENTS_QUERY, AppointmentPo.class);
		query.setParameter("patientIcn", patientICN);
		List<AppointmentPo> appointmentPos = query.getResultList();
		return appointmentPos;
		
	}
	
	private List<AppointmentPo> getPatientAppointments(String patientICN, Date startDate, Date endDate){
		
		final TypedQuery<AppointmentPo> query = this.entityManager.createQuery(APPOINTMENTS_BY_DATE_QUERY, AppointmentPo.class);
		query.setParameter("patientIcn", patientICN);
		query.setParameter("startDate", startDate);
		query.setParameter("endDate", endDate);
		List<AppointmentPo> appointmentPos = query.getResultList();
		return appointmentPos;
		
	}
	
	private AppointmentPo getPatientAppointment(String patientICN, String appointmentId){
		
		final TypedQuery<AppointmentPo> query = this.entityManager.createQuery(APPOINTMENT_QUERY, AppointmentPo.class);
		query.setParameter("patientIcn", patientICN);
		query.setParameter("id", appointmentId);
		AppointmentPo appointmentPo = query.getSingleResult();
		
		return appointmentPo;
		
	}

	private String getPatientICN (PatientIdentifier patientIdentifier) {
		
		PatientIdentityHelper helper = new PatientIdentityHelper();
		return helper.truncateICN(patientIdentifier).getUniqueId();
		
	}

	private Appointments translateToAppointments(List<? extends AppointmentPoBase> appointmentPos, PatientIdentifier patientIdentifier) {

		Appointments appointments = new Appointments();
		if(NullChecker.isNotNullish(appointmentPos)) {
			//appointments.setPatientIdentifier(patientIdentifier);
			for(AppointmentPoBase po : appointmentPos) {				
				appointments.add(po.translateToAppointment());
			}
			appointments.updateSystemId(SourceSystemCode.CDW);
		}
		return appointments;
	}

}
