package com.agilex.healthcare.hdr.appointment.model;

import com.agilex.soap.Message;
import org.joda.time.LocalDate;
import org.junit.*;
import org.w3c.dom.*;
import org.xml.sax.*;
import javax.xml.parsers.*;
import javax.xml.xpath.*;
import java.io.*;
import java.util.Date;
import static org.junit.Assert.*;

public class GetAppointmentsRequestTest {

    private static final String PATIENT_ID = "123";
    private static final String FILTER_START_DATE = "//entryPointFilter/startDate/text()";
    private static final String FILTER_END_DATE = "//entryPointFilter/endDate/text()";
    private static final String PATIENT_FILTER = "APPOINTMENTS_SINGLE_PATIENT_FILTER";
    private static final String HEALTH_ADAPTER = "HealthAdapter 1.0";

    private DocumentBuilder builder;

    @Before
    public void setUp() throws ParserConfigurationException {
        DocumentBuilderFactory domFactory = DocumentBuilderFactory.newInstance();
        domFactory.setNamespaceAware(true);
        builder = domFactory.newDocumentBuilder();
    }

    @Test
    public void requestMessageWithoutDateFilter() throws IOException, SAXException, XPathExpressionException {
        Message message = createFilterMessage(null, null);

        assertEquals("AppointmentsRead1", message.getBody().getRootElement().getChild("in0", null, null).getElementValue());
        assertEquals(PATIENT_FILTER, message.getBody().getRootElement().getChild("in2", null, null).getElementValue());
        assertEquals(HEALTH_ADAPTER, message.getBody().getRootElement().getChild("in3", null, null).getElementValue());

        Document doc = createFilterDoc(message);

        assertNull(getValue(FILTER_START_DATE, doc));
        assertNull(getValue(FILTER_END_DATE, doc));
    }

    @Test
    public void requestMessageWithStartDateFilter() throws IOException, SAXException, XPathExpressionException {
        Date startDate = new Date();
        Message message = createFilterMessage(startDate, null);

        Document doc = createFilterDoc(message);

        assertEquals(getDate(startDate), getValue(FILTER_START_DATE, doc));
        assertNull(getValue(FILTER_END_DATE, doc));
    }

    @Test
    public void requestMessageWithEndDateFilter() throws IOException, SAXException, XPathExpressionException {
        Date endDate = new Date();
        Message message = createFilterMessage(null, endDate);

        Document doc = createFilterDoc(message);

        assertNull(getValue(FILTER_START_DATE, doc));
        assertEquals(getDate(endDate), getValue(FILTER_END_DATE, doc));
    }

    @Test
    public void requestMessageWithStartEndDateFilter() throws IOException, SAXException, XPathExpressionException {
        Date startDate = new Date();
        Date endDate = new Date();
        Message message = createFilterMessage(startDate, endDate);

        Document doc = createFilterDoc(message);

        assertEquals(getDate(startDate), getValue(FILTER_START_DATE, doc));
        assertEquals(getDate(endDate), getValue(FILTER_END_DATE, doc));
    }

    private Message createFilterMessage(Date startDate, Date endDate) {
        GetAppointmentsRequest request = new GetAppointmentsRequest();
        return request.build(PATIENT_ID, startDate , endDate, HEALTH_ADAPTER);
    }

    private Document createFilterDoc(Message message) throws SAXException, IOException, XPathExpressionException {
        String filterXML = message.getBody().getRootElement().getChild("in1", null, null).getElementValue();

        InputSource is = new InputSource();
        is.setCharacterStream(new StringReader(filterXML));
        Document doc = builder.parse(is);


        assertEquals(PATIENT_FILTER, getValue("//filterId/text()", doc));
        assertEquals(HEALTH_ADAPTER, getValue("//clientName/text()", doc));
        assertEquals(PATIENT_ID, getValue("//patients/NationalId/text()", doc));
        assertEquals("Appointment", getValue("//entryPointFilter/domainEntryPoint/text()", doc));

        return doc;
    }

    private String getDate(Date date) {
        LocalDate localDate = new LocalDate(date);
        return localDate.toString("yyyy-MM-dd");
    }

    private String getValue(String xpathExp, Document doc) throws XPathExpressionException {
        XPathFactory factory = XPathFactory.newInstance();
        XPath xpath = factory.newXPath();
        XPathExpression expr = xpath.compile(xpathExp);

        Object result = expr.evaluate(doc, XPathConstants.NODESET);
        NodeList nodes = (NodeList) result;

        if (nodes.getLength() == 0)
            return null;
        else {
            return nodes.item(0).getNodeValue();
        }
    }
}
