package com.agilex.healthcare.hdr.appointment.model;

import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.w3c.dom.Node;

import com.agilex.healthcare.mobilehealthplatform.domain.Appointment;
import com.agilex.healthcare.mobilehealthplatform.domain.Appointments;
import com.agilex.healthcare.utility.XmlMapper;

public class AppointmentMapper extends XmlMapper {

    private static final String[] CLINIC_NAME_NODES = new String[]{"location", "identifier", "name"};
    private static final String[] CLINIC_PHONE_NODES = new String[]{"location", "telephone"};
    private static final String[] FACILITY_NAME_NODES = new String[]{"location", "institution", "identifier", "name"};
    private static final String[] APPT_STATUS_NODES = new String[]{"appointmentStatus", "displayText"};
    private static final String[] APPT_STATUS_CODE_NODES = new String[]{"appointmentStatus", "code"};
    private static final String[] APPT_ID_NODES = new String[]{"recordIdentifier", "identity"};
    private static final String[] APPT_ID_NAMESPACE_NODES = new String[]{"recordIdentifier", "namespaceId"};
    private static final String[] APPT_DATE_NODES = new String[]{"appointmentDateTime", "literal"};
    private static final DateTimeFormatter fmt = DateTimeFormat.forPattern("yyyyMMddHHmm");

    private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(AppointmentMapper.class);

    public AppointmentMapper(String responseXML) {
        super(responseXML, "appointment");
    }

    public Appointments map(String patientId) {
        Appointments appointments = new Appointments();

        for(int i=0; i < nodeList.getLength(); i++) {
            Appointment appointment = map(nodeList.item(i));
            appointment.setPatientId(patientId);

            appointments.add(appointment);
        }

        return appointments;
    }

    private Appointment map(Node node) {
        Appointment appointment = new Appointment();

        setUniqueId(node, appointment);
        setAppointmentStartDate(node, appointment);
        appointment.setClinicName(getNodeValue(node, CLINIC_NAME_NODES));
        appointment.setClinicPhone(getNodeValue(node, CLINIC_PHONE_NODES));
        appointment.setFacilityName(getNodeValue(node, FACILITY_NAME_NODES));
        appointment.setStatus(getNodeValue(node, APPT_STATUS_NODES));
        appointment.setStatusCode(getNodeValue(node,APPT_STATUS_CODE_NODES));

        return appointment;
    }

    private void setUniqueId(Node node, Appointment appointment) {
        String recordId = getNodeValue(node, APPT_ID_NODES);
        String recordNamespace = getNodeValue(node, APPT_ID_NAMESPACE_NODES);

        try {
            appointment.setUniqueId(recordId + "_" + recordNamespace);
        } catch (Exception ex) {
            logger.warn("Unable to generate appointment id from recordId/ recordNamespace.", ex);
        }
    }

    private void setAppointmentStartDate(Node node, Appointment appointment) {
        String apptDateTime = getNodeValue(node, APPT_DATE_NODES);

        if (apptDateTime == null)
            return;

        try {
            appointment.setAppointmentStartDate(fmt.parseDateTime(apptDateTime).toDate());
        } catch (Exception ex) {
            logger.warn("Unable to parse appointment date/time: " + apptDateTime, ex);
        }
    }
}
