package com.agilex.healthcare.appointment.datalayer;

import static junit.framework.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import org.junit.Before;
import org.junit.Test;

import com.agilex.healthcare.hdr.appointment.datalayer.AppointmentRepository;
import com.agilex.healthcare.hdr.config.HDRConfig;
import com.agilex.healthcare.mobilehealthplatform.domain.Appointment;
import com.agilex.healthcare.mobilehealthplatform.domain.Appointments;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.system.health.SmokeTest;
import com.agilex.system.health.SmokeTestStatus;

public class AppointmentRepositoryTest {

    AppointmentRepository repository;

    @Before
    public void init() {
    	Properties p = new Properties();
	    String HDR_ENDPOINT_URL = "";

	    try {
        // note: properties file is loading from the classpath; if the int tests are being run through gradle,
        // this file can be placed at the root + /intTest/spring/properties/va-veteran-dev.int.properties
	    	InputStream is = ClassLoader.getSystemResourceAsStream("spring/properties/va-veteran-dev.int.properties");
	    	p.load(is);
	    	HDR_ENDPOINT_URL = p.getProperty("hdr.endPointUrl");
	    } catch (IOException e) {
	    	HDR_ENDPOINT_URL = "http://localhost:8080/hdr/mockPathwaysHttpBinding";
	    }
//        HDRConfig config = new HDRConfig("http://localhost:8080/hdr/mockPathwaysHttpBinding", 5000, 30000,"HealthAdapter 1.0", "R,XYZ");
        HDRConfig config = new HDRConfig(HDR_ENDPOINT_URL, 5000, 30000, "HealthAdapter 1.0", "R");
        repository = new AppointmentRepository(config);
    }

    @Test
    public void getAllAppointments() {
        Appointments appointments = repository.fetchAppointmentsByPatient(new PatientIdentifier("mock-aa", "D123401"), DateFilterFactory.createEmptyFilter());
        assertTrue(appointments.size()>0);

        for (Appointment appointment : appointments) {
			if("R".equals(appointment.getStatusCode()) == false){
				fail("No show or cancelled appointments found");
			}
		}
    }

    @Test
    public void getAllAppointmentsNoneFound() {
        Appointments appointments = repository.fetchAppointmentsByPatient(new PatientIdentifier("mock-aa", "PATID40"), DateFilterFactory.createEmptyFilter());

        assertEquals(0, appointments.size());
    }

    @Test
    public void getAppointment() {
        Appointment appointment = repository.fetchAppointment(new PatientIdentifier("mock-aa", "123"), "3110830.13_544_2.98");

        assertEquals("123", appointment.getPatientId());
        assertEquals("3110830.13_544_2.98", appointment.getUniqueId());
        assertEquals("Tue Aug 30 13:00:00 EDT 2011", appointment.getAppointmentStartDate().toString());
        assertEquals("SCHEDULED/KEPT", appointment.getStatus());
        assertEquals("WJB G DENTAL HYGIENIST", appointment.getClinicName());
        assertEquals("800-293-8262", appointment.getClinicPhone());
        assertEquals("GREENVILLE SOC", appointment.getFacilityName());
    }

    @Test
    public void smokeTestSuccess() {
        SmokeTest test = repository;
        SmokeTestStatus status = test.run();

        assertTrue(status.isSuccess());
    }


    @Test
    public void smokeTestFailure() {
        HDRConfig config = new HDRConfig("http://hadev.invalid.com:8089/hdr/mockPathwaysHttpBinding", 100, 30000, "HealthAdapter 1.0");
        SmokeTest test = new AppointmentRepository(config);
        SmokeTestStatus status = test.run();

        assertFalse(status.isSuccess());
        assertEquals("java.net.SocketTimeoutException", status.getExceptionType());
    }

}
