package com.agilex.healthcare.adr.model;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import junit.framework.Assert;

import org.junit.Before;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.utility.NullSafeStringComparer;

public class DemographicMapperTest {

	DemographicMapper mapper;

	public DemographicMapperTest() {
	}

	@Before
	public void before() {
		String responseXml = "<SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\"><SOAP-ENV:Header/><SOAP-ENV:Body><getEESummaryResponse xmlns=\"http://jaxws.webservices.esr.domain.ext/schemas\"><eesVersion>1.4.0.01001</eesVersion><summary><sensitivityInfo><sensityFlag>false</sensityFlag></sensitivityInfo><associations><association><familyName>BARREE</familyName><givenName>MERLE</givenName><contactType>Primary Next of Kin</contactType><primaryPhone>(111)111-1111</primaryPhone></association><association><familyName>BARREE</familyName><givenName>MERLE</givenName><contactType>Designee</contactType><primaryPhone>(111)111-1111</primaryPhone></association><association><familyName>BARREE</familyName><givenName>MERLE</givenName><contactType>Emergency Contact</contactType><primaryPhone>(111)111-1111</primaryPhone></association></associations><demographics><contactInfo><phones><phone><phoneNumber>(333)333-3333</phoneNumber><type>Home</type></phone><phone><phoneNumber>(111)111-1111</phoneNumber><type>Business</type></phone></phones><addresses><address><addressTypeCode>Permanent</addressTypeCode><addressChangeDateTime>2009-11-19T08:14:24.000-06:00</addressChangeDateTime><addressChangeSource>Veterans Affairs Medical Center</addressChangeSource><addressChangeSite>442 - CHEYENNE VAMC</addressChangeSite><zipCode>82009</zipCode><state>WY</state><country>USA</country><city>CHEYENNE</city><county>LARAMIE</county><line1>4608 MORAN AVE APT C</line1></address></addresses></contactInfo></demographics></summary><invocationDate>2012-08-13T10:37:43.785-05:00</invocationDate></getEESummaryResponse></SOAP-ENV:Body></SOAP-ENV:Envelope>";
		mapper = new DemographicMapper(responseXml);
	}

	@Test
	public void verifyMappingOfPhones() {
		PatientDemographics demographics = mapper.map();
		assertNotNull(demographics.getPhoneNumbers());
		assertEquals(2, demographics.getPhoneNumbers().size());
		
		assertTrue(containsPhoneNumber(demographics.getPhoneNumbers(), "(333)333-3333"));
		assertTrue(containsPhoneNumber(demographics.getPhoneNumbers(), "(111)111-1111"));
		
		assertEquals("(333)333-3333", demographics.getPhoneNumberHome());
		assertEquals("(111)111-1111", demographics.getPhoneNumberWork());
		
		assertEquals(1, demographics.getAddresses().size());
		
		assertTrue(containsAddress(demographics.getAddresses(), "4608 MORAN AVE APT C", "CHEYENNE, WY 82009"));
		
		assertEquals("4608 MORAN AVE APT C", demographics.getAddress().getStreetAddressLine1());
		assertEquals("CHEYENNE, WY 82009", demographics.getAddress().getStreetAddressLine3());
		
		assertNotNull(demographics.getNextOfKin());
		assertEquals("BARREE, MERLE", demographics.getNextOfKin().getName());
		assertEquals("Primary Next of Kin", demographics.getNextOfKin().getRelationship());
		assertEquals("(111)111-1111", demographics.getNextOfKin().getPhoneNumber());
	}
	
	@Test
	public void verifyMappingOfAddresses() {
		PatientDemographics demographics = mapper.map();
		assertNotNull(demographics.getAddresses());
		assertEquals(1, demographics.getAddresses().size());
		
		assertTrue(containsAddress(demographics.getAddresses(), "4608 MORAN AVE APT C", "CHEYENNE, WY 82009"));
		
		assertEquals("4608 MORAN AVE APT C", demographics.getAddress().getStreetAddressLine1());
		assertEquals("CHEYENNE, WY 82009", demographics.getAddress().getStreetAddressLine3());
	}
	
	@Test
	public void verifyAssociations() {
		String responseXml = "<SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\"><SOAP-ENV:Header/><SOAP-ENV:Body><getEESummaryResponse xmlns=\"http://jaxws.webservices.esr.domain.ext/schemas\"><eesVersion>1.4.0.01001</eesVersion><summary><sensitivityInfo><sensityFlag>false</sensityFlag></sensitivityInfo><associations><association><address><city>Chantilly</city><country>US</country><county>Fairfax</county><line1>5155 Parkstone Dr</line1><state>VA</state><zipCode>20152</zipCode><zipPlus4>6648</zipPlus4></address><alternatePhone>7037747227</alternatePhone><contactType>Primary Next Of Kin</contactType><familyName>Allen</familyName><givenName>Wright</givenName><lastUpdateDate>2013-03-12</lastUpdateDate><middleName>A</middleName><organizationName>TEST</organizationName><primaryPhone>7030000000</primaryPhone><relationship>Father</relationship></association></associations><demographics><contactInfo><phones><phone><phoneNumber>(333)333-3333</phoneNumber><type>Home</type></phone><phone><phoneNumber>(111)111-1111</phoneNumber><type>Business</type></phone></phones><addresses><address><addressTypeCode>Permanent</addressTypeCode><addressChangeDateTime>2009-11-19T08:14:24.000-06:00</addressChangeDateTime><addressChangeSource>Veterans Affairs Medical Center</addressChangeSource><addressChangeSite>442 - CHEYENNE VAMC</addressChangeSite><zipCode>82009</zipCode><state>WY</state><country>USA</country><city>CHEYENNE</city><county>LARAMIE</county><line1>4608 MORAN AVE APT C</line1></address></addresses></contactInfo></demographics></summary><invocationDate>2012-08-13T10:37:43.785-05:00</invocationDate></getEESummaryResponse></SOAP-ENV:Body></SOAP-ENV:Envelope>";
		PatientDemographics demographics = new DemographicMapper(responseXml).map();
		NextOfKin nextOfKin = demographics.getNextOfKin();
		Assert.assertNotNull(nextOfKin);
		
		Assert.assertEquals("5155 Parkstone Dr", nextOfKin.getStreetAddressLine1());
		Assert.assertEquals("Chantilly, VA 20152", nextOfKin.getStreetAddressLine3());
	}
	
	@Test
	public void verifyParsingWhenAssociationsDoesNotExist(){
		String responseXml = "<SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\"><SOAP-ENV:Header/><SOAP-ENV:Body><getEESummaryResponse xmlns=\"http://jaxws.webservices.esr.domain.ext/schemas\"><eesVersion>1.4.0.01001</eesVersion><summary><sensitivityInfo><sensityFlag>false</sensityFlag></sensitivityInfo><demographics><contactInfo><phones><phone><phoneNumber>(333)333-3333</phoneNumber><type>Home</type></phone><phone><phoneNumber>(111)111-1111</phoneNumber><type>Business</type></phone></phones><addresses><address><addressTypeCode>Permanent</addressTypeCode><addressChangeDateTime>2009-11-19T08:14:24.000-06:00</addressChangeDateTime><addressChangeSource>Veterans Affairs Medical Center</addressChangeSource><addressChangeSite>442 - CHEYENNE VAMC</addressChangeSite><zipCode>82009</zipCode><state>WY</state><country>USA</country><city>CHEYENNE</city><county>LARAMIE</county><line1>4608 MORAN AVE APT C</line1></address></addresses></contactInfo></demographics></summary><invocationDate>2012-08-13T10:37:43.785-05:00</invocationDate></getEESummaryResponse></SOAP-ENV:Body></SOAP-ENV:Envelope>";
		DemographicMapper mapper = new DemographicMapper(responseXml);
		PatientDemographics demographics = mapper.map();
		assertTrue(containsAddress(demographics.getAddresses(), "4608 MORAN AVE APT C", "CHEYENNE, WY 82009"));
		assertEquals("4608 MORAN AVE APT C", demographics.getAddress().getStreetAddressLine1());
		assertEquals("CHEYENNE, WY 82009", demographics.getAddress().getStreetAddressLine3());
	}
	
	@Test
	public void validateSuccessWhenNoPhone() {
		String responseXml = "<SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\"><SOAP-ENV:Header/><SOAP-ENV:Body><getEESummaryResponse xmlns=\"http://jaxws.webservices.esr.domain.ext/schemas\"><eesVersion>1.4.0.01001</eesVersion><summary><sensitivityInfo><sensityFlag>false</sensityFlag></sensitivityInfo><associations><association><familyName>BARREE</familyName><givenName>MERLE</givenName><contactType>Primary Next of Kin</contactType><primaryPhone>(111)111-1111</primaryPhone></association><association><familyName>BARREE</familyName><givenName>MERLE</givenName><contactType>Designee</contactType><primaryPhone>(111)111-1111</primaryPhone></association><association><familyName>BARREE</familyName><givenName>MERLE</givenName><contactType>Emergency Contact</contactType><primaryPhone>(111)111-1111</primaryPhone></association></associations><demographics><contactInfo><addresses><address><addressTypeCode>Permanent</addressTypeCode><addressChangeDateTime>2009-11-19T08:14:24.000-06:00</addressChangeDateTime><addressChangeSource>Veterans Affairs Medical Center</addressChangeSource><addressChangeSite>442 - CHEYENNE VAMC</addressChangeSite><zipCode>82009</zipCode><state>WY</state><country>USA</country><city>CHEYENNE</city><county>LARAMIE</county><line1>4608 MORAN AVE APT C</line1></address></addresses></contactInfo></demographics></summary><invocationDate>2012-08-13T10:37:43.785-05:00</invocationDate></getEESummaryResponse></SOAP-ENV:Body></SOAP-ENV:Envelope>";
		DemographicMapper mapper = new DemographicMapper(responseXml);
		PatientDemographics demographics = mapper.map();
		
		assertTrue(containsAddress(demographics.getAddresses(), "4608 MORAN AVE APT C", "CHEYENNE, WY 82009"));
		assertEquals("4608 MORAN AVE APT C", demographics.getAddress().getStreetAddressLine1());
		assertEquals("CHEYENNE, WY 82009", demographics.getAddress().getStreetAddressLine3());
		
		NextOfKin nextOfKin = demographics.getNextOfKin();
		Assert.assertNotNull(nextOfKin);
		assertEquals("BARREE, MERLE", demographics.getNextOfKin().getName());
		assertEquals("Primary Next of Kin", demographics.getNextOfKin().getRelationship());
		assertEquals("(111)111-1111", demographics.getNextOfKin().getPhoneNumber());
	}
	
	@Test
	public void validateSuccessWhenNoPhoneOrAddress() {
		String responseXml = "<SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\"><SOAP-ENV:Header/><SOAP-ENV:Body><getEESummaryResponse xmlns=\"http://jaxws.webservices.esr.domain.ext/schemas\"><eesVersion>1.4.0.01001</eesVersion><summary><sensitivityInfo><sensityFlag>false</sensityFlag></sensitivityInfo><associations><association><familyName>BARREE</familyName><givenName>MERLE</givenName><contactType>Primary Next of Kin</contactType><primaryPhone>(111)111-1111</primaryPhone></association><association><familyName>BARREE</familyName><givenName>MERLE</givenName><contactType>Designee</contactType><primaryPhone>(111)111-1111</primaryPhone></association><association><familyName>BARREE</familyName><givenName>MERLE</givenName><contactType>Emergency Contact</contactType><primaryPhone>(111)111-1111</primaryPhone></association></associations><demographics><contactInfo></contactInfo></demographics></summary><invocationDate>2012-08-13T10:37:43.785-05:00</invocationDate></getEESummaryResponse></SOAP-ENV:Body></SOAP-ENV:Envelope>";
		DemographicMapper mapper = new DemographicMapper(responseXml);
		PatientDemographics demographics = mapper.map();
		
		NextOfKin nextOfKin = demographics.getNextOfKin();
		Assert.assertNotNull(nextOfKin);
		assertEquals("BARREE, MERLE", demographics.getNextOfKin().getName());
		assertEquals("Primary Next of Kin", demographics.getNextOfKin().getRelationship());
		assertEquals("(111)111-1111", demographics.getNextOfKin().getPhoneNumber());
	}
	
	@Test
	public void validateSuccessWithNoPrimaryNextOfKin() {
		String responseXml = "<SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http://schemas.xmlsoap.org/soap/envelope/\"><SOAP-ENV:Header/><SOAP-ENV:Body><getEESummaryResponse xmlns=\"http://jaxws.webservices.esr.domain.ext/schemas\"><eesVersion>1.4.0.01001</eesVersion><summary><sensitivityInfo><sensityFlag>false</sensityFlag></sensitivityInfo><associations><association><familyName>BARREE</familyName><givenName>MERLE</givenName><contactType>Designee</contactType><primaryPhone>(111)111-1111</primaryPhone></association><association><familyName>BARREE</familyName><givenName>MERLE</givenName><contactType>Emergency Contact</contactType><primaryPhone>(111)111-1111</primaryPhone></association></associations><demographics><contactInfo><phones><phone><phoneNumber>(333)333-3333</phoneNumber><type>Home</type></phone><phone><phoneNumber>(111)111-1111</phoneNumber><type>Business</type></phone></phones><addresses><address><addressTypeCode>Permanent</addressTypeCode><addressChangeDateTime>2009-11-19T08:14:24.000-06:00</addressChangeDateTime><addressChangeSource>Veterans Affairs Medical Center</addressChangeSource><addressChangeSite>442 - CHEYENNE VAMC</addressChangeSite><zipCode>82009</zipCode><state>WY</state><country>USA</country><city>CHEYENNE</city><county>LARAMIE</county><line1>4608 MORAN AVE APT C</line1></address></addresses></contactInfo></demographics></summary><invocationDate>2012-08-13T10:37:43.785-05:00</invocationDate></getEESummaryResponse></SOAP-ENV:Body></SOAP-ENV:Envelope>";
		DemographicMapper mapper = new DemographicMapper(responseXml);
		PatientDemographics demographics = mapper.map();
		
		NextOfKin nextOfKin = demographics.getNextOfKin();
		Assert.assertNotNull(nextOfKin);
		Assert.assertEquals("Designee", nextOfKin.getRelationship());
	}

    @Test
    public void validateSuccessfulParsing() {
        String responseXml = "<SOAP-ENV:Envelope xmlns:SOAP-ENV=\"http:schemas.xmlsoap.org/soap/envelope/\"><SOAP-ENV:Header/><SOAP-ENV:Body><getEESummaryResponse><summary><associations><association><contactType>Phone</contactType><familyName>MICKEY-MOUSE</familyName><givenName>zzJACQUELINEMORGAN</givenName><primaryPhone>(307)555-7740</primaryPhone><relationship>HUSBAND</relationship></association></associations><demographics><contactInfo><addresses><address><city>CHEYENNE</city><country>US</country><line1>C/O R MARTIN (11B), 2360 E PERSHING BLVD</line1><state>WYOMING</state><zipCode>82001</zipCode><phoneNumber>(307)555-7739</phoneNumber></address></addresses><phones><phone><phoneNumber>(307)555-7743</phoneNumber><type>Work</type></phone><phone><phoneNumber>(307)555-7744</phoneNumber><type>Home</type></phone><phone><phoneNumber>(307)555-7745</phoneNumber><type>Mobile</type></phone></phones></contactInfo></demographics></summary></getEESummaryResponse></SOAP-ENV:Body></SOAP-ENV:Envelope>";
        DemographicMapper mapper = new DemographicMapper(responseXml);
        PatientDemographics demographics = mapper.map();
    }
	
	boolean containsPhoneNumber(PhoneNumbers phoneNumbers, String value) {
		for(PhoneNumber phoneNumber : phoneNumbers) {
			if(NullSafeStringComparer.areEqual(phoneNumber.getNumber(), value)) {
				return true;
			}
		}
		return false;
	}
	
	boolean containsAddress(Addresses addresses, String line1, String line3) {
		for(Address address : addresses) {
			if(NullSafeStringComparer.areEqual(address.getStreetAddressLine1(), line1) 
					&& NullSafeStringComparer.areEqual(address.getStreetAddressLine3(), line3)) {
				return true;
			}
		}
		return false;
	}

}
