package com.agilex.healthcare.adr.model;

import java.util.ArrayList;
import java.util.List;

import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.agilex.healthcare.adr.model.code.DemographicCode;
import com.agilex.healthcare.mobilehealthplatform.domain.Address;
import com.agilex.healthcare.mobilehealthplatform.domain.Addresses;
import com.agilex.healthcare.mobilehealthplatform.domain.NextOfKin;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientDemographics;
import com.agilex.healthcare.mobilehealthplatform.domain.PhoneNumber;
import com.agilex.healthcare.mobilehealthplatform.domain.PhoneNumbers;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.XmlMapper;

public class DemographicMapper extends XmlMapper {

	private static final String[] ADDRESS_INFO_ELEMENT = new String[] { "address" };
	private static final String[] PHONE_NUMBER_NODES = new String[] { "phoneNumber" };
	private static final String[] PHONE_TYPE_NODES = new String[] { "type" };
	private static final String[] ADDRESS_LINE1_NODES = new String[] { "line1" };
	private static final String[] ADDRESS_LINE2_NODES = new String[] { "line2" };
	private static final String[] ADDRESS_LINE3_NODES = new String[] { "line3" };
	private static final String[] ADDRESS_CITY_NODES = new String[] { "city" };
	private static final String[] ADDRESS_STATE_NODES = new String[] { "state" };
	private static final String[] ADDRESS_ZIP_NODES = new String[] { "zipCode" };
	private static final String[] ADDRESS_TYPE_NODES = new String[] { "addressTypeCode" };
	private static final String[] KIN_FIRST_NAME_NODES = new String[] { "givenName" };
	private static final String[] KIN_LAST_NAME_NODES = new String[] { "familyName" };
	private static final String[] KIN_CONTACT_TYPE = new String[] { "contactType" };
    private static final String[] KIN_RELATIONSHIP = new String[] { "relationship" };
    private static final String[] KIN_PRIMARY_PHONE = new String[] { "primaryPhone" };
	private static final String[] KIN_ALTERNATE_PHONE = new String[] { "alternatePhone" };
	private static final String[] BASE_ADDRESS_INFO = new String[] { "baseAddressInfo" };

	public DemographicMapper(String responseXml) {
		super(responseXml, "summary");
	}

	public PatientDemographics map() {
		if (nodeList == null || nodeList.getLength() == 0)
			return null;

		NodeList childNodes = nodeList.item(0).getChildNodes();
		NodeList demographicNodes = null;
		NodeList associationNodes = null;
		for (int i = 0; i < childNodes.getLength(); i++) {
			Element item = (Element) childNodes.item(i);
			if ("demographics".equalsIgnoreCase(item.getNodeName())) {
				demographicNodes = item.getChildNodes();
			} else if ("associations".equalsIgnoreCase(item.getNodeName())) {
				associationNodes = item.getChildNodes();
			}
		}

		PatientDemographics demographics = map(demographicNodes, associationNodes);

		return demographics;
	}

	PatientDemographics map(NodeList demographicNodes, NodeList kinAssociations) {
		if (demographicNodes == null)
			return null;

		PatientDemographics demographics = new PatientDemographics();

		for (int i = 0; i < demographicNodes.getLength(); i++) {
			Node contactInfo = demographicNodes.item(i);
			demographics = map(contactInfo, kinAssociations);
		}

		return demographics;
	}

	/**
	 * 	<xs:complexType name="associationCollection">
			<xs:all>
				<xs:element maxOccurs="1" minOccurs="0" name="association" nillable="true" type="associationInfo"/>
			</xs:all>
		</xs:complexType>
	 * <xs:complexType name="associationInfo">
			<xs:all>
				x <xs:element minOccurs="0" name="address" type="baseAddressInfo"/>
				<xs:element minOccurs="0" name="alternatePhone" type="xs:string"/>
				x <xs:element minOccurs="0" name="contactType" type="xs:string"/>
				x <xs:element minOccurs="0" name="familyName" type="xs:string"/>
				x <xs:element minOccurs="0" name="givenName" type="xs:string"/>
				<xs:element minOccurs="0" name="lastUpdateDate" type="xs:dateTime"/>
				<xs:element minOccurs="0" name="middleName" type="xs:string"/>
				<xs:element minOccurs="0" name="organizationName" type="xs:string"/>
				<xs:element minOccurs="0" name="prefix" type="xs:string"/>
				x <xs:element minOccurs="0" name="primaryPhone" type="xs:string"/>
				<xs:element minOccurs="0" name="relationship" type="xs:string"/>
				<xs:element minOccurs="0" name="suffix" type="xs:string"/>
			</xs:all>
	  *	</xs:complexType>
	 */
	List<NextOfKin> mapAssociationCollection(NodeList associations) {
		List<NextOfKin> possibleNextOfKin = new ArrayList<NextOfKin>();
		
		if(associations != null) {
			for(int i = 0; i < associations.getLength(); i++) {
				NextOfKin association = new NextOfKin();
				
				String firstName = getNodeValue(associations.item(i), KIN_FIRST_NAME_NODES);
				String lastName = getNodeValue(associations.item(i), KIN_LAST_NAME_NODES);
				
				association.setName(lastName + ", " + firstName);
				association.setPhoneNumber(getNodeValue(associations.item(i), KIN_PRIMARY_PHONE));
				association.setRelationship(getNodeValue(associations.item(i), KIN_RELATIONSHIP));
				association.setPhoneNumberWork(getNodeValue(associations.item(i), KIN_ALTERNATE_PHONE));
				
				Element item = (Element) associations.item(i);
				NodeList addresses = item.getElementsByTagName("address");
				if(addresses != null && addresses.getLength() > 0) {
					Node addressNode = addresses.item(0);
					Address address = parseAddressInfo(addressNode);
					if (address != null) {
						association.setStreetAddressLine1(address.getStreetAddressLine1());
						association.setStreetAddressLine2(address.getStreetAddressLine2());
						association.setStreetAddressLine3(address.getStreetAddressLine3());
						association.setCity(address.getCity());
						association.setState(address.getState());
					}
				}
				
				possibleNextOfKin.add(association);
			}
		}
		
		return possibleNextOfKin;
	}
	
	PatientDemographics map(Node contactInfo, NodeList kinAssociations) {
		PatientDemographics demographics = new PatientDemographics();

		Element contact = (Element) contactInfo;

		PhoneNumbers phoneNumbers = parsePhones(contact);
		demographics.setPhoneNumbers(phoneNumbers);
		setPhoneNumbers(phoneNumbers, demographics);

		Addresses addresses = parseAddresses(contact);
		demographics.setAddresses(addresses);
		demographics.setAddress(findPermanent(addresses));

		List<NextOfKin> kins = mapAssociationCollection(kinAssociations);
		NextOfKin primaryKin = findKinByType(kins, DemographicCode.KIN_PRIMARY_NEXT_OF_KIN);
		if(primaryKin == null && NullChecker.isNotNullish(kins)) {
			primaryKin = kins.get(0);
		}
		demographics.setNextOfKin(primaryKin);

		return demographics;
	}

	private PhoneNumbers parsePhones(Element demographics) {
		PhoneNumbers phoneNumbers = new PhoneNumbers();
		try {
			Element phones = getNextElement(demographics, "phones");
			NodeList phoneNodes = phones.getChildNodes();
			for (int i = 0; i < phoneNodes.getLength(); i++) {
				PhoneNumber phoneNumber = new PhoneNumber();
				phoneNumber.setNumber(getNodeValue(phoneNodes.item(i), PHONE_NUMBER_NODES));
				phoneNumber.setType(getNodeValue(phoneNodes.item(i), PHONE_TYPE_NODES));

				phoneNumbers.add(phoneNumber);
			}
		} catch (Exception ex) {
			// phone number not available
		}
		return phoneNumbers;
	}

	private Addresses parseAddresses(Element demographics) {
		Addresses addresses = new Addresses();

		try {
			Element addressesElement = getNextElement(demographics, "addresses");
			if (addressesElement != null) {
				NodeList addressNodes = addressesElement.getChildNodes();
				for (int i = 0; i < addressNodes.getLength(); i++) {
					Address address = parseAddressInfo(addressNodes.item(i));

					addresses.add(address);
				}
			}
		} catch (Exception ex) {
			// addresses not available
		}

		return addresses;
	}
	
	Address parseAddressInfo(Node addressNode) {
		Address address = new Address();
		address.setStreetAddressLine1(getNodeValue(addressNode, ADDRESS_LINE1_NODES));
		address.setAddressType(getNodeValue(addressNode, ADDRESS_TYPE_NODES));
		
		String line2 = getNodeValue(addressNode, ADDRESS_LINE2_NODES);
		String line3 = getNodeValue(addressNode, ADDRESS_LINE3_NODES);
		String city = getNodeValue(addressNode, ADDRESS_CITY_NODES);
		String state = getNodeValue(addressNode, ADDRESS_STATE_NODES);
		String zipCode = getNodeValue(addressNode, ADDRESS_ZIP_NODES);

		address.setStreetAddressLine2(line2);
		
		// This can be a bit confusing - ADR provides 3 separate elements that can make up "line 3" and they also provide
		// a line3 element. We do a check to see if one is more reliable than the other and pray for the best.
		if((NullChecker.isNullish(city) || NullChecker.isNullish(state) || NullChecker.isNullish(zipCode))) {
			address.setStreetAddressLine3(line3);
		} else {
			address.setStreetAddressLine3(String.format("%s, %s %s", city, state, zipCode));
		}
		
		return address;
	}

	Address findPermanent(Addresses addresses) {
		Address addressMatch = null;
		if (NullChecker.isNotNullish(addresses))
			addressMatch = addresses.get(0);
		else
			return addressMatch;

		for (Address address : addresses) {
			if (DemographicCode.ADDRESS_PERMANENT.equals(address.getAddressType())) {
				addressMatch = address;
			}
		}
		return addressMatch;
	}

	NextOfKin findKinByType(List<NextOfKin> kins, String kinType) {
		NextOfKin kinMatch = null;
		if (NullChecker.isNullish(kins))
			return kinMatch;

		for (NextOfKin nextOfKin : kins) {
			if (kinType.equals(nextOfKin.getRelationship())) {
				kinMatch = nextOfKin;
			}
		}

		return kinMatch;
	}

	void setPhoneNumbers(PhoneNumbers phoneNumbers, PatientDemographics demographics) {
		if (NullChecker.isNullish(phoneNumbers))
			return;

		for (PhoneNumber phoneNumber : phoneNumbers) {

			if (DemographicCode.PHONE_BUSINESS.equalsIgnoreCase(phoneNumber.getType()) ||
                    DemographicCode.PHONE_BUSINESS2.equalsIgnoreCase(phoneNumber.getType())) {
				demographics.setPhoneNumberWork(phoneNumber.getNumber());
			} else if (DemographicCode.PHONE_PAGER.equalsIgnoreCase(phoneNumber.getType())) {
				demographics.setPhoneNumberPager(phoneNumber.getNumber());
			} else if (DemographicCode.PHONE_HOME.equalsIgnoreCase(phoneNumber.getType())) {
				demographics.setPhoneNumberHome(phoneNumber.getNumber());
			} else if (DemographicCode.PHONE_MOBILE.equalsIgnoreCase(phoneNumber.getType())) {
				demographics.setPhoneNumberMobile(phoneNumber.getNumber());
			} else if (DemographicCode.PHONE_PAGER.equalsIgnoreCase(phoneNumber.getType())) {
                demographics.setPhoneNumberPager(phoneNumber.getNumber());
            }

		}
	}
}
