package com.agilex.healthcare.mobilehealthplatform.security;

import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.userdetails.UserDetails;

import com.agilex.healthcare.mobilehealthplatform.datalayer.patient.AgeCalculator;
import com.agilex.healthcare.mobilehealthplatform.domain.DOBDate;
import com.agilex.healthcare.mobilehealthplatform.domain.MhpUser;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientData;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.security.oauth.IamAppUser;
import com.agilex.security.oauth.IamUser;

public class MhpUserFactory {
    private static final String DSLOGON_DATE_FORMAT = "yyyy-MM-dd";
	private static final String SSOE_ASSIGNING_AUTHORITY = "EDIPI";

	public static MhpUser createFromSecurityContext() {
		
        SecurityContext ctx = SecurityContextHolder.getContext();

        return createFromAuthentication(ctx.getAuthentication());
    }

    public static MhpUser createFromAuthentication(Authentication authentication) {
    	if (authentication == null) {
    		return null;
    	}
    	
        if (authentication.getPrincipal() instanceof IamAppUser)
            return getMhpUserFromIamUser(authentication);
        else if (authentication.getPrincipal() instanceof AppUser)
            return getMhpUserFromAppUser(authentication);
        else
            return null;
    }

    private static MhpUser getMhpUserFromAppUser(Authentication authentication) {
        AppUser appUser = (AppUser) authentication.getPrincipal();

        MhpUser mhpUser = appUser.getMhpUser();
        mhpUser.setUserName(appUser.getUsername());

        return mhpUser;
    }

    private static MhpUser getMhpUserFromIamUser(Authentication authentication1) {
        IamAppUser appUser = (IamAppUser) authentication1.getPrincipal();
        IamUser iamUser = appUser.getIamUser();

        MhpUser mhpUser = new MhpUser();
        PatientIdentifier userIdentifier = new PatientIdentifier(SSOE_ASSIGNING_AUTHORITY, iamUser.getId());
        mhpUser.setUserIdentifier(userIdentifier);

        mhpUser.setFirstName(iamUser.getFirstName());
        mhpUser.setLastName(iamUser.getLastName());
        mhpUser.setMiddleInitials(iamUser.getMiddleName());
        mhpUser.setAccessCode(iamUser.getAccessCode());
        
        DOBDate dslogonDateOfBirth = new DOBDate(DSLOGON_DATE_FORMAT, getDatePart(iamUser.getPatientDOB()));
        String vistaDateOfBirth = DateHelper.format(dslogonDateOfBirth.getDate(), DateHelper.REPORT_DATE_FORMAT);
        mhpUser.getPatient().setDateOfBirth(new DOBDate(DateHelper.REPORT_DATE_FORMAT, vistaDateOfBirth));
        mhpUser.getPatient().setGender(iamUser.getPatientGender());
        mhpUser.getPatient().setLastName(iamUser.getPatientLastName());
        mhpUser.getPatient().setFirstName(iamUser.getPatientFirstName());
        mhpUser.getPatient().setMiddleName(iamUser.getPatientMiddleName());
        mhpUser.getPatient().setSsn(iamUser.getPatientSSN());
        mhpUser.getPatient().setAge(Long.toString(AgeCalculator.calculateAge(DateHelper.parse(getDatePart(iamUser.getPatientDOB()), DSLOGON_DATE_FORMAT))));
        mhpUser.setUserName(appUser.getUsername());

        PatientIdentifier patientIdentifier = new PatientIdentifier(SSOE_ASSIGNING_AUTHORITY, iamUser.getPatientEdipId());
        mhpUser.getPatient().setPatientIdentifier(patientIdentifier);
        mhpUser.setRightOfAccessAccepted(iamUser.IsRightOfAccessAccepted());

        return mhpUser;
    }

    private static String getDatePart(String dob) {
        int dateEnds = dob.indexOf("T");

        if (dateEnds != -1)
            return dob.substring(0, dateEnds);
        else
            return dob;
    }

	public static Set<String> getCurrentUserRoles() {

		SecurityContext securityContext = SecurityContextHolder.getContext();

        Authentication auth = securityContext.getAuthentication();
        UserDetails userDetails = (UserDetails) auth.getPrincipal();

        Set<String> roles = new HashSet<String>();

        Collection<? extends GrantedAuthority> grantedRoles = userDetails.getAuthorities();

        for (GrantedAuthority grantedAuthority : grantedRoles) {
			roles.add(grantedAuthority.getAuthority());
		}

		return roles;
	}

	public static boolean doesUserHasStaffRole() {
		boolean staff = false;

		if (getCurrentUserRoles().contains(Roles.ROLE_STAFF)) {
			staff = true;
		}

		return staff;
	}

    public static void addSurrogateToPatientData(PatientData data) {
        if (SecurityContextHolder.getContext().getAuthentication() == null)
            return;

            MhpUser user = createFromSecurityContext();
        if (user != null && !user.userIsPatient()){
        	data.setSurrogateIdentifier("EDIPI", user.getId());
        }
    }

}
