package com.agilex.healthcare.mobilehealthplatform.security;

import java.util.ArrayList;
import java.util.List;

import javax.ws.rs.core.MediaType;

import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.oauth2.common.OAuth2AccessToken;
import org.springframework.security.oauth2.provider.AuthorizationRequest;
import org.springframework.security.oauth2.provider.DefaultAuthorizationRequest;
import org.springframework.security.oauth2.provider.OAuth2Authentication;
import org.springframework.security.oauth2.provider.token.ResourceServerTokenServices;

import com.agilex.healthcare.mobilehealthplatform.domain.MhpUser;
import com.agilex.healthcare.mobilehealthplatform.dto.ClientAuthorizationRequest;
import com.agilex.healthcare.mobilehealthplatform.dto.HAOauth2Authentication;
import com.agilex.healthcare.mobilehealthplatform.dto.TokenValidationRequest;
import com.agilex.healthcare.utility.NullChecker;
import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.UniformInterfaceException;

public class HATokenValidation implements ResourceServerTokenServices {
	
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(HATokenValidation.class);

	private String tokenValidationUri;

	public HATokenValidation(String tokenValidationUri) {
		this.tokenValidationUri = tokenValidationUri;
	}

	@Override
	public OAuth2Authentication loadAuthentication(String accessToken) throws AuthenticationException {

		OAuth2Authentication oauth2Authentication = retrieveAndConstructOauth2Authentication(accessToken);
		return oauth2Authentication;
	}

	private OAuth2Authentication retrieveAndConstructOauth2Authentication(String accessToken) {
		OAuth2Authentication oauth2Authentication;

		TokenValidationRequest tokenValidationRequest = createTokenValidationRequest(accessToken);
		
		HAOauth2Authentication haOauth2Authentication = null;
		
		try {
			 haOauth2Authentication = validateToken(tokenValidationRequest);
		}catch(UniformInterfaceException e){
			if (e.getResponse().getStatus() == 204){
				logger.info("Invalid token");
				return null;
			}else{
				logger.info("Exception", e);
				throw new RuntimeException(e);
			}
		}

		oauth2Authentication = constructOauth2Authentication(haOauth2Authentication);

		return oauth2Authentication;
	}

	private OAuth2Authentication constructOauth2Authentication(HAOauth2Authentication haOauth2Authentication) {
		OAuth2Authentication oauth2Authentication;
		AuthorizationRequest authorizationRequest = createAuthorizationRequest(haOauth2Authentication);

		MhpUser mhpUser = haOauth2Authentication.getMhpUser();
		
		List<GrantedAuthority> authorities = createGrantedAuthorities(haOauth2Authentication.getAuthorities());
		
		AppUser appUser = new AppUser(haOauth2Authentication.getUserName(), "N/A", authorities, mhpUser);

		UsernamePasswordAuthenticationToken userAuthentication = new UsernamePasswordAuthenticationToken(appUser, "N/A", authorities);

		oauth2Authentication = new OAuth2Authentication(authorizationRequest, userAuthentication);
		return oauth2Authentication;
	}

	private List<GrantedAuthority> createGrantedAuthorities(List<String> authorities) {
		
		List<GrantedAuthority> grantedAuthorities = new ArrayList<GrantedAuthority>();

		if(NullChecker.isNullish(authorities)){
			return grantedAuthorities;
		}
		
		
		for (String authority : authorities) {
			grantedAuthorities.add(new SimpleGrantedAuthority(authority));
		}

		return grantedAuthorities;
	}

	private HAOauth2Authentication validateToken(TokenValidationRequest tokenValidationRequest) {
		Client restClient = new com.sun.jersey.api.client.Client();
		HAOauth2Authentication haOauth2Authentication = null;
		haOauth2Authentication = restClient.resource(tokenValidationUri).accept(MediaType.APPLICATION_JSON).post(HAOauth2Authentication.class,tokenValidationRequest);

		return haOauth2Authentication;
	}

	private TokenValidationRequest createTokenValidationRequest(String accessToken) {
		TokenValidationRequest tokenValidationRequest = new TokenValidationRequest();
		tokenValidationRequest.setToken(accessToken);
		return tokenValidationRequest;
	}

	private AuthorizationRequest createAuthorizationRequest(HAOauth2Authentication haOauth2Authentication) {
		ClientAuthorizationRequest clientAuthorizationRequest = haOauth2Authentication.getAuthorizatioRequest();
		DefaultAuthorizationRequest authorizationRequest = new DefaultAuthorizationRequest(clientAuthorizationRequest.getClientId(),clientAuthorizationRequest.getScope());
		authorizationRequest.setApproved(clientAuthorizationRequest.isApproved());
		authorizationRequest.setRedirectUri(clientAuthorizationRequest.getRedirectUri());
		authorizationRequest.setResourceIds(clientAuthorizationRequest.getResourceIds());
		
		return authorizationRequest;
	}

	@Override
	public OAuth2AccessToken readAccessToken(String accessToken) {
		throw new UnsupportedOperationException();
	}
}
