package com.agilex.healthcare.mobilehealthplatform.oauth;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Map;

import javax.sql.DataSource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.codehaus.jackson.map.ObjectMapper;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.security.oauth2.common.exceptions.InvalidClientException;
import org.springframework.security.oauth2.provider.ClientDetails;
import org.springframework.security.oauth2.provider.JdbcClientDetailsService;
import org.springframework.security.oauth2.provider.NoSuchClientException;
import org.springframework.util.Assert;


public class OauthJdbcClientDetailsService extends JdbcClientDetailsService {

    private static final Log logger = LogFactory.getLog(OauthJdbcClientDetailsService.class);

    private static final String CLIENT_FIELDS_FOR_UPDATE = "resource_ids, scope, "
                    + "authorized_grant_types, web_server_redirect_uri, authorities, access_token_validity, "
                    + "refresh_token_validity, timeout_value, long_durable_session, additional_information ";

    private static final String CLIENT_FIELDS = "client_secret, " + CLIENT_FIELDS_FOR_UPDATE;

    private static final String BASE_FIND_STATEMENT = "select client_id, " + CLIENT_FIELDS
                    + " from oauth_client_details";

    private static final String DEFAULT_SELECT_STATEMENT = BASE_FIND_STATEMENT + " where client_id = ?";

    private static final String UPDATE_TIME_OUT_VALUE_STATEMENT = "update oauth_client_details "
            + "set timeout_value = ? where client_id = ?";
    
    private RowMapper<ClientDetails> oauthRowMapper = new OauthClientDetailsRowMapper();

    private String selectClientDetailsSql = DEFAULT_SELECT_STATEMENT;
    private String updateClientTimeoutSql = UPDATE_TIME_OUT_VALUE_STATEMENT;

    private final JdbcTemplate jdbcTemplate;

    public OauthJdbcClientDetailsService(DataSource dataSource) {
        super(dataSource);
        Assert.notNull(dataSource, "DataSource required");
        this.jdbcTemplate = new JdbcTemplate(dataSource);
    }
    

    public ClientDetails loadClientByClientId(String clientId) throws InvalidClientException {
        ClientDetails details;
        try {
                details = jdbcTemplate.queryForObject(selectClientDetailsSql, new OauthClientDetailsRowMapper(), clientId);
        }
        catch (EmptyResultDataAccessException e) {
                throw new NoSuchClientException("No client with requested id");
        }
        return details;
    }

    public void updateClientTimeout(String clientId, String timeoutValue) throws NoSuchClientException {
        int count = jdbcTemplate.update(updateClientTimeoutSql, timeoutValue, clientId);
        if (count != 1) {
                throw new NoSuchClientException("No client found with id");
        }
    }
    
    public void setSelectClientDetailsSql(String selectClientDetailsSql) {
        this.selectClientDetailsSql = selectClientDetailsSql;
    }

    public void setUpdateClientTimeoutSql(String updateClientTimeoutSql) {
        this.updateClientTimeoutSql = updateClientTimeoutSql;
    }
    
    /**
     * @param oauthRowMapper the oauthRowMapper to set
     */
    public void setOauthRowMapper(RowMapper<ClientDetails> oauthRowMapper) {
            this.oauthRowMapper = oauthRowMapper;
    }

    private static class OauthClientDetailsRowMapper implements RowMapper<ClientDetails> {
        private ObjectMapper mapper = new ObjectMapper();

        public OauthClientDetails mapRow(ResultSet rs, int rowNum) throws SQLException {
            OauthBaseClientDetails details = new OauthBaseClientDetails(rs.getString(1), rs.getString(3), rs.getString(4),
                                rs.getString(5), rs.getString(7), rs.getString(6));
                details.setClientSecret(rs.getString(2));
                if (rs.getObject(8) != null) {
                        details.setAccessTokenValiditySeconds(rs.getInt(8));
                }
                if (rs.getObject(9) != null) {
                        details.setRefreshTokenValiditySeconds(rs.getInt(9));
                }
                if (rs.getObject(10) != null) {
                    details.setTimeoutSeconds(rs.getInt(10));
                }
                if (rs.getObject(11) != null) {
                    details.setLongDurableSession(rs.getBoolean(11));
                }
                String json = rs.getString(12);
                if (json != null) {
                        try {
                                @SuppressWarnings("unchecked")
                                Map<String, Object> additionalInformation = mapper.readValue(json, Map.class);
                                details.setAdditionalInformation(additionalInformation);
                        }
                        catch (Exception e) {
                                logger.warn("Could not decode JSON for additional information");
                        }
                }
                return details;
        }
    }

}
