package com.agilex.healthcare.mobilehealthplatform.authorization;

import java.util.Collection;
import java.util.Date;

import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.MediaType;

import org.springframework.security.access.AccessDecisionVoter;
import org.springframework.security.access.ConfigAttribute;
import org.springframework.security.core.Authentication;
import org.springframework.security.web.FilterInvocation;

import com.agilex.healthcare.mobilehealthplatform.oauth.Oauth2AccessTokenReader;
import com.agilex.healthcare.utility.DateHelper;
import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.ClientHandlerException;
import com.sun.jersey.api.client.ClientRequest;
import com.sun.jersey.api.client.ClientResponse;
import com.sun.jersey.api.client.filter.ClientFilter;

public class TimeoutVoter<S> implements AccessDecisionVoter<S> {
	
	private int applicationTimeoutInSeconds = 900;
	private String resourceLastAccessedTimeUrl = "http://localhost:8080/AuthorizationServices/rest/resourceLastAccessedTime";

	public TimeoutVoter(String lastAccessedTimeResourceUrl, int timeout) {
		resourceLastAccessedTimeUrl = lastAccessedTimeResourceUrl;
		applicationTimeoutInSeconds = timeout;
	}

	@Override
	public boolean supports(ConfigAttribute attribute) {
        return "CAN_TIMEOUT".equals(attribute.getAttribute());
    }

	@Override
	public boolean supports(Class<?> clazz) {
		return true;
	}

	@Override
	public int vote(Authentication authentication, S s, Collection<ConfigAttribute> attributes) {
        for (ConfigAttribute attribute : attributes) {
            if (supports(attribute)){
                if (isUserWithinTimeoutBoundary(authentication, s))
                    return AccessDecisionVoter.ACCESS_GRANTED;
                else
                    return AccessDecisionVoter.ACCESS_DENIED;
            }
        }
        return AccessDecisionVoter.ACCESS_ABSTAIN;
	}

	private boolean isUserWithinTimeoutBoundary(Authentication authentication, S s) {
		if (applicationTimeoutInSeconds > DateHelper.calculateDeltaInSeconds(getLastAccessedTime(s), new Date())) {
			setResourceLastAccessedTime(s);
			return true;
		} else {
			return false;
		}
	}
	
	private Date getLastAccessedTime(S s){
	
		ResourceLastAccessedTime resourceLastAccessedTime = fetchResourceLastAccessedTime(s);

		if(resourceLastAccessedTime != null){
			return resourceLastAccessedTime.getLastAccessedTime();
		}else{
			return new Date();
		}
	}

	private ResourceLastAccessedTime fetchResourceLastAccessedTime(S s){
		Client restClient = createRestClientWithAuthorizationHeader(s);
		ResourceLastAccessedTime resourceLastAccessedTime = null;
		
		try {
			resourceLastAccessedTime = getResourceLastAccessedTimeFromAuthorizationServices(restClient);
		} catch (Exception e) {
			// Ignore the authorization services communication errors 	
			resourceLastAccessedTime = null;
		}

		return resourceLastAccessedTime;
	}

	protected ResourceLastAccessedTime getResourceLastAccessedTimeFromAuthorizationServices(Client restClient) {
		ResourceLastAccessedTime resourceLastAccessedTime = null;
		resourceLastAccessedTime = restClient.resource(resourceLastAccessedTimeUrl).accept(MediaType.APPLICATION_JSON).get(ResourceLastAccessedTime.class);
		return resourceLastAccessedTime;
	}

	private Client createRestClientWithAuthorizationHeader(S s) {
		HttpServletRequest request = ((FilterInvocation) s).getHttpRequest();

		Oauth2AccessTokenReader reader = new Oauth2AccessTokenReader();
		String token = reader.readTokenFromHttpRequest(request);
		final String authHeader = "Bearer " + token;
		
		Client restClient = new com.sun.jersey.api.client.Client();
		restClient.addFilter(new ClientFilter() {
			
		    @Override
		    public ClientResponse handle(final ClientRequest cr) throws ClientHandlerException {

		        if (!cr.getHeaders().containsKey(HttpHeaders.AUTHORIZATION)) {
		            cr.getHeaders().add(HttpHeaders.AUTHORIZATION, authHeader);
		        }
		        return getNext().handle(cr);
		    }
		});
		return restClient;
	}

	//TODO: Is this blocking call required as part of authorization? May be this should be asynchronous
	private void setResourceLastAccessedTime(S s) {
		Client restClient = createRestClientWithAuthorizationHeader(s);

		try {
			saveResourceLastAccessedTimeThroughAuthorizationServices(restClient);
		} catch (Exception e) {
			// Ignore the authorization services communication errors 	
		}
	}

	protected void saveResourceLastAccessedTimeThroughAuthorizationServices(Client restClient) {
		restClient.resource(resourceLastAccessedTimeUrl).accept(MediaType.APPLICATION_JSON).post(ResourceLastAccessedTime.class);
	}
	
}
