package com.agilex.healthcare.mobilehealthplatform.authorization;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;

import junit.framework.Assert;

import org.junit.Test;
import org.springframework.mock.web.MockHttpServletRequest;
import org.springframework.mock.web.MockHttpServletResponse;
import org.springframework.security.access.AccessDecisionVoter;
import org.springframework.security.access.ConfigAttribute;
import org.springframework.security.access.SecurityConfig;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.oauth2.provider.DefaultAuthorizationRequest;
import org.springframework.security.oauth2.provider.OAuth2Authentication;
import org.springframework.security.web.FilterInvocation;

import com.agilex.healthcare.mobilehealthplatform.domain.MhpUser;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.security.AppUser;
import com.agilex.healthcare.mobilehealthplatform.security.Roles;
import com.sun.jersey.api.client.Client;

public class TimeoutVoterTest{
    private String timeOutSupportedAttribute = "CAN_TIMEOUT";	
	private String userName = "noidentity.user";
	private String assigningAuthority = "EDIPI";
	private String uniqueId = "noidentity";
	private String lastAccessedTimeResourceUrl = "http://authservcies.com/rest/api";

    private static final int TIME_OUT_IN_SECONDS = 5;

    protected boolean supportsConfigAttribute(String config, AccessDecisionVoter<FilterInvocation> decisionVoter) {
		AccessDecisionVoter<FilterInvocation> voter = decisionVoter;
		ConfigAttribute configAttribute = new SecurityConfig(config);
		return voter.supports(configAttribute);
	}
    
	protected FilterInvocation createFilterInvocation(String assigningAuthority, String patientId) {
		String servletPath = "/MobileHealthPlatformWeb";
		String pathInfo = "/rest/patient/" + assigningAuthority + "/" + patientId;

        ServletRequest request = createServletRequest(servletPath, pathInfo);
        ServletResponse response = createServletResponse();
        FilterChain chain = createFilterChain();
        FilterInvocation fi = new FilterInvocation(request, response, chain);
		return fi;
	}
 
    static final FilterChain DUMMY_CHAIN = new FilterChain() {
        public void doFilter(ServletRequest req, ServletResponse res) throws IOException, ServletException {
            throw new UnsupportedOperationException("Dummy filter chain");
        }
    };

    private FilterChain createFilterChain() {
        return DUMMY_CHAIN;
    }

    private ServletResponse createServletResponse() {
        return new MockHttpServletResponse();
    }

    private ServletRequest createServletRequest(String servletPath, String pathInfo) {
        ServletRequest request = new MockHttpServletRequest(null, servletPath + pathInfo);
        return request;  
    }
	
	
	@Test
	public void supportsCanTimeOutConfigAccess(){
		Assert.assertTrue(supportsConfigAttribute("CAN_TIMEOUT", new TimeoutVoter<FilterInvocation>(lastAccessedTimeResourceUrl, TIME_OUT_IN_SECONDS)));
	}

	@Test
	public void doesNotSupportsInvalidConfigAccess(){
		Assert.assertFalse(supportsConfigAttribute(Roles.ROLE_CONSUMER, new TimeoutVoter<FilterInvocation>(lastAccessedTimeResourceUrl, TIME_OUT_IN_SECONDS)));
	}
	
	@Test
	public void testAccessDeniedWhenTimeoutExceedsLimit() throws Exception {
		TimeoutVoter<FilterInvocation> timeoutVoter = createTimeOutVoter();
		Authentication userAuthentication = createUserAuthentication();
		
		FilterInvocation fi = createFilterInvocation(assigningAuthority, uniqueId);
		
		Collection<ConfigAttribute> attributes = createSupportedAttributes();
		Thread.sleep(6000);
		int result = timeoutVoter.vote(userAuthentication, fi, attributes);
		
        Assert.assertEquals(AccessDecisionVoter.ACCESS_DENIED, result);
	}

	@Test
	public void testAccessAbstainedWhenUnsupportedAttributesAreSpecified() throws Exception {
		TimeoutVoter<FilterInvocation> timeoutVoter = createTimeOutVoter();
		Authentication userAuthentication = createUserAuthentication();
		
		FilterInvocation fi = createFilterInvocation(assigningAuthority, uniqueId);
		
		Collection<ConfigAttribute> attributes = createUnSupportedAttributes();
		Thread.sleep(6000);
		int result = timeoutVoter.vote(userAuthentication, fi, attributes);
		
        Assert.assertEquals(AccessDecisionVoter.ACCESS_ABSTAIN, result);
	}
	
	@Test
	public void testAccessDeniedWhenTimeoutExceedsLimitAndUsingOauthAuthentication() throws Exception {
		TimeoutVoter<FilterInvocation> timeoutVoter = createTimeOutVoter();
		Authentication userAuthentication = createUserAuthentication();
		Authentication oauthAuthentication = createOauth2Authentication(userAuthentication);
		
		FilterInvocation fi = createFilterInvocation(assigningAuthority, uniqueId);
		
		Collection<ConfigAttribute> attributes = createSupportedAttributes();
		Thread.sleep(6000);
		int result = timeoutVoter.vote(oauthAuthentication, fi, attributes);
		
        Assert.assertEquals(AccessDecisionVoter.ACCESS_DENIED, result);
	}
	
	
	@Test
	public void testAccessGrantedWhenTimeoutDoesNotExceedLimit() throws Exception {
		TimeoutVoter<FilterInvocation> timeoutVoter = createTimeOutVoter();
		Authentication userAuthentication = createUserAuthentication();
		
		FilterInvocation fi = createFilterInvocation(assigningAuthority, uniqueId);
		
		Collection<ConfigAttribute> attributes = createSupportedAttributes();
		Thread.sleep(4000);
		int result = timeoutVoter.vote(userAuthentication, fi, attributes);
		
        Assert.assertEquals(AccessDecisionVoter.ACCESS_GRANTED, result);
	}

	@Test
	public void testAccessGrantedWhenTheThereisNoResourceLastAccessedTime() throws Exception {
		TimeoutVoter<FilterInvocation> timeoutVoter = new TimeoutVoter<FilterInvocation>(lastAccessedTimeResourceUrl, TIME_OUT_IN_SECONDS){

			@Override
			protected ResourceLastAccessedTime getResourceLastAccessedTimeFromAuthorizationServices(Client restClient) {
				return null;
			}
			
			protected void saveResourceLastAccessedTimeThroughAuthorizationServices(Client restClient) {
			}	
		};

		Authentication userAuthentication = createUserAuthentication();
		
		FilterInvocation fi = createFilterInvocation(assigningAuthority, uniqueId);
		
		Collection<ConfigAttribute> attributes = createSupportedAttributes();
		int result = timeoutVoter.vote(userAuthentication, fi, attributes);
		
        Assert.assertEquals(AccessDecisionVoter.ACCESS_GRANTED, result);
	}

	@Test
	public void testAccessGrantedWhenThereisNoAuthorizationServices() throws Exception {
		TimeoutVoter<FilterInvocation> timeoutVoter = new TimeoutVoter<FilterInvocation>(lastAccessedTimeResourceUrl, TIME_OUT_IN_SECONDS){

			@Override
			protected ResourceLastAccessedTime getResourceLastAccessedTimeFromAuthorizationServices(Client restClient) {
				throw new RuntimeException();
			}
			
			protected void saveResourceLastAccessedTimeThroughAuthorizationServices(Client restClient) {
				throw new RuntimeException();
			}	
		};

		Authentication userAuthentication = createUserAuthentication();
		
		FilterInvocation fi = createFilterInvocation(assigningAuthority, uniqueId);
		
		Collection<ConfigAttribute> attributes = createSupportedAttributes();
		int result = timeoutVoter.vote(userAuthentication, fi, attributes);
		
        Assert.assertEquals(AccessDecisionVoter.ACCESS_GRANTED, result);
	}
	
	@Test
	public void testAccessGrantedWhenAuthorizationServicesExist() throws Exception {
		TimeoutVoter<FilterInvocation> timeoutVoter = new TimeoutVoter<FilterInvocation>(lastAccessedTimeResourceUrl, TIME_OUT_IN_SECONDS);

		Authentication userAuthentication = createUserAuthentication();
		
		FilterInvocation fi = createFilterInvocation(assigningAuthority, uniqueId);
		
		Collection<ConfigAttribute> attributes = createSupportedAttributes();
		int result = timeoutVoter.vote(userAuthentication, fi, attributes);
		
        Assert.assertEquals(AccessDecisionVoter.ACCESS_GRANTED, result);
	}

	private Authentication createUserAuthentication(){
		List<GrantedAuthority> authorities = createPatientAuthorities();

		AppUser principal = createPrincipal(userName, new PatientIdentifier(assigningAuthority, uniqueId), authorities);
		setPermittedPatient(principal, assigningAuthority, uniqueId);
		
		Object credentials = null;

		Authentication authentication = createAuthentication(authorities, principal, credentials);
		return authentication;
	}
	
	private TimeoutVoter<FilterInvocation> createTimeOutVoter() {
		TimeoutVoter<FilterInvocation> timeoutVoter = new TimeoutVoter<FilterInvocation>(lastAccessedTimeResourceUrl, TIME_OUT_IN_SECONDS){
			final Date lastAccessedTime = new Date();

			@Override
			protected ResourceLastAccessedTime getResourceLastAccessedTimeFromAuthorizationServices(Client restClient) {
				ResourceLastAccessedTime resourceLastAccessedTime = new ResourceLastAccessedTime();
				resourceLastAccessedTime.setLastAccessedTime(lastAccessedTime);
				resourceLastAccessedTime.setUserId(userName);
				return resourceLastAccessedTime;
			}
			
			protected void saveResourceLastAccessedTimeThroughAuthorizationServices(Client restClient) {
			}	
		};

		return timeoutVoter;
	}

	
    private Collection<ConfigAttribute> createSupportedAttributes() {
        Collection<ConfigAttribute> attributes = new ArrayList<ConfigAttribute>();
        ConfigAttribute attribute = new ConfigAttribute() {
			private static final long serialVersionUID = 9208629368448580596L;

			@Override
            public String getAttribute() {
                return timeOutSupportedAttribute;
            }
        };                                             
        attributes.add(attribute);
        return attributes;
    }

    private Collection<ConfigAttribute> createUnSupportedAttributes() {
        Collection<ConfigAttribute> attributes = new ArrayList<ConfigAttribute>();
        ConfigAttribute attribute = new ConfigAttribute() {
			private static final long serialVersionUID = 9208629368448580596L;

			@Override
            public String getAttribute() {
                return "ROLE_CONSUMER";
            }
        };                                             
        attributes.add(attribute);
        return attributes;
    }
 
    private void setPermittedPatient(AppUser principal, String defaultAssigningAuthority1, String targetPatientId) {
        Patient advocatePatientTarget = new Patient();
        PatientIdentifier patientIdentifier = new PatientIdentifier();
        patientIdentifier.setAssigningAuthority(defaultAssigningAuthority1);
        patientIdentifier.setUniqueId(targetPatientId);
        advocatePatientTarget.setPatientIdentifier(patientIdentifier);
        principal.getMhpUser().setPatient(advocatePatientTarget);
    }
    
    private OAuth2Authentication createOauth2Authentication(Authentication userAuthentication){
    	return new OAuth2Authentication(new DefaultAuthorizationRequest(
    			Collections.<String, String> emptyMap()), userAuthentication);
    }
    
	protected List<GrantedAuthority> createPatientAuthorities() {
		return createAuthorities(Roles.ROLE_CONSUMER);
	}
	
	protected List<GrantedAuthority> createAuthorities(String role) {
		List<GrantedAuthority> authorities = new ArrayList<GrantedAuthority>();
		authorities.add(new SimpleGrantedAuthority(role));
		return authorities;
	}

	protected AppUser createPrincipal(String userName, PatientIdentifier patientIdentifier, List<GrantedAuthority> authorities) {
		String password = "";
		MhpUser authenticatedUser = new MhpUser();
        authenticatedUser.setUserIdentifier(patientIdentifier);
        authenticatedUser.setRightOfAccessAccepted(true);
		
		AppUser principal = new AppUser(userName, password, authorities, authenticatedUser);
		return principal;
	}
	
    protected UsernamePasswordAuthenticationToken createAuthentication(List<GrantedAuthority> authorities, AppUser principal, Object credentials) {
        return new UsernamePasswordAuthenticationToken(principal, credentials, authorities);
    }
}
