package com.agilex.healthcare.mobilehealthplatform.oauth;

import java.util.ArrayList;
import java.util.Date;

import javax.annotation.Resource;
import javax.ws.rs.Consumes;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Response;

import org.springframework.context.annotation.Scope;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.oauth2.common.exceptions.InvalidTokenException;
import org.springframework.security.oauth2.provider.AuthorizationRequest;
import org.springframework.security.oauth2.provider.OAuth2Authentication;
import org.springframework.security.oauth2.provider.token.ResourceServerTokenServices;
import org.springframework.stereotype.Service;

import com.agilex.healthcare.mobilehealthplatform.authorization.ResourceLastAccessedTime;
import com.agilex.healthcare.mobilehealthplatform.authorization.ResourceLastAccessedTimeDAO;
import com.agilex.healthcare.mobilehealthplatform.domain.MhpUser;
import com.agilex.healthcare.mobilehealthplatform.dto.ClientAuthorizationRequest;
import com.agilex.healthcare.mobilehealthplatform.dto.HAOauth2Authentication;
import com.agilex.healthcare.mobilehealthplatform.dto.TokenValidationRequest;
import com.agilex.healthcare.mobilehealthplatform.security.AppUser;
import com.agilex.healthcare.mobilehealthplatform.security.MhpUserFactory;
import com.agilex.healthcare.mobilehealthplatform.security.Roles;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.security.oauth.IamAppUser;
import com.agilex.security.oauth.TokenStoreLogout;

@Path("/validateToken")
@Service
@Scope("request")
public class Oauth2AuthenticationResource {
	
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(Oauth2AuthenticationResource.class);

	@Resource
	private ResourceServerTokenServices tokenServices;
	
	@Resource
	private TokenStoreLogout tokenStore;

	@Resource
	private ResourceLastAccessedTimeDAO resourceLastAccessedTimeDAO;

	@Resource
	private OauthJdbcClientDetailsService oauthJdbcClientDetailsService;
	
	@Resource(name = "sessionTimeOutInSeconds")
	private int sessionTimeOutInSeconds = 900;

	@POST
	@Consumes({"application/xml", "application/json"})
	@Produces({"application/xml", "application/json" })
	public Response validateToken(TokenValidationRequest tokenValidationRequest) {

		if(isInvalidInput(tokenValidationRequest)){
			return createBadRequestResponse();
		}
		
		OAuth2Authentication authentication = loadAuthenticationForGivenToken(tokenValidationRequest);		 
		
		if (authentication != null){
		        MhpUser mhpUser = getUser(authentication);
		        Date  date = getLastAccessedTime(mhpUser); 
			if (isRequestWithinTimeoutBoundary(mhpUser, date)) {
			    HAOauth2Authentication haOauthAuthentication = createHAOauth2Authentication(authentication);
			    return createResponseWithAuthentication(haOauthAuthentication);
			} else {
			    tokenStore.removeAllUSerTokens(authentication);
			} 
		}
		
		return null;
	}

        private boolean isRequestWithinTimeoutBoundary(MhpUser mhpUser, Date date) {
		
		if (sessionTimeOutInSeconds > DateHelper.calculateDeltaInSeconds(date, new Date())) {
			setResourceLastAccessedTime(mhpUser);
			return true;
		} else {
			logger.info("Request is not within timeout boundary");
			return false;
		}
	}

        private Date getLastLoginTime(MhpUser mhpUser){
            ResourceLastAccessedTime resourceLastAccessedTime = resourceLastAccessedTimeDAO.fetchResourceLastAccessedTime(mhpUser.getUserIdentifier().toString());
            return resourceLastAccessedTime.getLastLoginTime();
        }
        
	private Date getLastAccessedTime(MhpUser mhpUser){
		ResourceLastAccessedTime resourceLastAccessedTime = resourceLastAccessedTimeDAO.fetchResourceLastAccessedTime(mhpUser.getUserIdentifier().toString());
		return resourceLastAccessedTime.getLastAccessedTime();
	}
	
	private void setResourceLastAccessedTime(MhpUser mhpUser){
		ResourceLastAccessedTime resourceLastAccessedTime = new ResourceLastAccessedTime();
		resourceLastAccessedTime.setLastAccessedTime(new Date());
		resourceLastAccessedTime.setUserId(mhpUser.getUserIdentifier().toString());
		resourceLastAccessedTimeDAO.saveResourceLastAccessedTime(resourceLastAccessedTime);
	}
	
	private MhpUser getUser(OAuth2Authentication authentication){
		MhpUser mhpUser = MhpUserFactory.createFromAuthentication(authentication.getUserAuthentication());
		return mhpUser;
	}
	
	private Response createResponseWithAuthentication(
			HAOauth2Authentication haOauthAuthentication) {
		return Response.status(200).entity(haOauthAuthentication).build();
	}

	private HAOauth2Authentication createHAOauth2Authentication(OAuth2Authentication authentication) {
		HAOauth2Authentication haOauthAuthentication = new HAOauth2Authentication();

		setClientAuthorizationRequest(authentication.getAuthorizationRequest(), haOauthAuthentication);
		setAuthoritiesandUserName(authentication, haOauthAuthentication);
		setMhpUser(authentication, haOauthAuthentication);
		setLastLoginTime(haOauthAuthentication);

		return haOauthAuthentication;
	}

	private void setLastLoginTime(HAOauth2Authentication haOauthAuthentication) {
		ResourceLastAccessedTime resourceLastAccessedTime = resourceLastAccessedTimeDAO.fetchResourceLastAccessedTime(haOauthAuthentication.getMhpUser().getUserIdentifier().toString());
		haOauthAuthentication.setLastLoginDate(resourceLastAccessedTime.getLastLoginTime());
	}

	private void setMhpUser(OAuth2Authentication authentication, HAOauth2Authentication haOauthAuthentication) {
		haOauthAuthentication.setMhpUser(getUser(authentication));
	}

	//TODO: Remove 2 users objects
	private void setAuthoritiesandUserName(OAuth2Authentication authentication, HAOauth2Authentication haOauthAuthentication) {
		ArrayList<String> authorities = new ArrayList<String>();

		if (authentication.getUserAuthentication().getPrincipal() instanceof AppUser) {
			AppUser appUser = (AppUser) authentication.getUserAuthentication().getPrincipal();
			haOauthAuthentication.setUserName(appUser.getUsername());

			authorities.add(Roles.ROLE_MHP_USER);
			authorities.add(Roles.ROLE_STAFF);

		} else if (authentication.getUserAuthentication().getPrincipal() instanceof IamAppUser) {
			IamAppUser iamAppUser = (IamAppUser) authentication.getUserAuthentication().getPrincipal();
			haOauthAuthentication.setUserName(iamAppUser.getUsername());

			authorities.add(Roles.ROLE_MHP_USER);
			authorities.add(Roles.ROLE_CONSUMER);
		}
		
		haOauthAuthentication.setAuthorities(authorities);
	}

	private Response createBadRequestResponse() {
		return Response.status(400).entity("Invalid Token Validation Request").build();
	}

	protected OAuth2Authentication loadAuthenticationForGivenToken(TokenValidationRequest tokenValidationRequest) {
		OAuth2Authentication authentication = null;
		
		try {
			authentication = tokenServices.loadAuthentication(tokenValidationRequest.getToken());
		} catch (AuthenticationException authenticationException) {
			logger.info("Either token was expired or not available", authenticationException);
		} catch (InvalidTokenException invalidTokenException) {
			logger.info("Either token was expired or not available", invalidTokenException);
		}

		return authentication;
	}

	private boolean isInvalidInput(TokenValidationRequest tokenValidationRequest) {
		return tokenValidationRequest == null || tokenValidationRequest.getToken() == null;
	}

	private void setClientAuthorizationRequest(AuthorizationRequest authorizationReuest, HAOauth2Authentication haOauth2Authentication) {
		ClientAuthorizationRequest clientAuthorizationRequest = new ClientAuthorizationRequest();

		clientAuthorizationRequest.setApprovalParameters(authorizationReuest
				.getApprovalParameters());
		clientAuthorizationRequest.setClientId(authorizationReuest.getClientId());
		clientAuthorizationRequest.setScope(authorizationReuest.getScope());
		clientAuthorizationRequest.setResourceIds(authorizationReuest.getResourceIds());
		clientAuthorizationRequest.setApproved(authorizationReuest.isApproved());
		clientAuthorizationRequest.setDenied(authorizationReuest.isDenied());
		clientAuthorizationRequest.setState(authorizationReuest.getState());
		clientAuthorizationRequest.setRedirectUri(authorizationReuest.getRedirectUri());
		clientAuthorizationRequest.setResponseTypes(authorizationReuest.getResponseTypes());

		haOauth2Authentication.setAuthorizatioRequest(clientAuthorizationRequest);
	}

}
