package com.agilex.healthcare.mobilehealthplatform.auth.provider.mdws;

import static java.util.concurrent.TimeUnit.SECONDS;

import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;

import javax.xml.namespace.QName;
import javax.xml.ws.BindingProvider;

import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

import com.agilex.healthcare.mobilehealthplatform.authorization.ResourceLastAccessedTime;
import com.agilex.healthcare.mobilehealthplatform.authorization.ResourceLastAccessedTimeDAO;
import com.agilex.healthcare.mobilehealthplatform.domain.MhpUser;
import com.agilex.healthcare.mobilehealthplatform.mdws.MdwsInfo;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.EmrSvc;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.EmrSvcSoap;
import com.agilex.healthcare.mobilehealthplatform.oauth.StaffAppAuthenticator;
import com.agilex.healthcare.mobilehealthplatform.security.VistaAuthenticator;

public class MdwsAppAuthenticator extends VistaAuthenticator implements StaffAppAuthenticator, ApplicationContextAware {
	
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(MdwsAppAuthenticator.class);
	
	private static ScheduledExecutorService scheduler;
	public static Map<String, ScheduledFuture<?>> scheduledTasks = Collections.synchronizedMap(new HashMap<String, ScheduledFuture<?>>());
	public static Map<String, List<String>> userSessionMap = Collections.synchronizedMap(new HashMap<String, List<String>>());

	private ApplicationContext springContext;
	private int sessionTimeOutInSeconds = 900;
	private int mdwsKeepAliveIntervalInSeconds = 90;
	
	public MdwsAppAuthenticator(String baseEmrServiceUrl, String emrServiceUrl, int scheduledThreadPool, int sessionTimeOutInSeconds, int mdwsKeepAliveIntervalInSeconds) {
		super(baseEmrServiceUrl, emrServiceUrl);
		MdwsAppAuthenticator.scheduler = Executors.newScheduledThreadPool(scheduledThreadPool);
		this.sessionTimeOutInSeconds = sessionTimeOutInSeconds;
		this.mdwsKeepAliveIntervalInSeconds = mdwsKeepAliveIntervalInSeconds;
	}
	
    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        springContext = applicationContext;
    }
	
	@Override
	public MhpUser authenticate(String accessCode, String verifyCode, String siteCode) throws Exception {
		logger.info("MdwsAppAuthenticator - Authenticating user for the site");
		MhpUser mhpUser = super.authenticate(accessCode, verifyCode, siteCode);
		resetResourceLastAccessedTime(mhpUser, new Date());
		if (!userSessionMap.containsKey(mhpUser.getUserIdentifier().toString())) {
			keepConnectionActive(mhpUser);
		} else {
			mhpUser.getMdwsInfo().setSessionId(userSessionMap.get(mhpUser.getUserIdentifier().toString()).get(0));
		}
		logger.info("MdwsAppAuthenticator - User is Authenticated for the site");
		return mhpUser;
	}
	
	private void resetResourceLastAccessedTime(MhpUser mhpUser, Date lastAccessedTime) {
		ResourceLastAccessedTime resourceLastAccessedTime = new ResourceLastAccessedTime();
		resourceLastAccessedTime.setUserId(mhpUser.getUserIdentifier().toString());
		resourceLastAccessedTime.setLastAccessedTime(lastAccessedTime);
		resourceLastAccessedTime.setLastLoginTime(lastAccessedTime);
		ResourceLastAccessedTimeDAO dao = springContext.getBean(ResourceLastAccessedTimeDAO.class);
		logger.debug("MdwsAppAuthenticator - Storing user last accessed time");
		dao.saveResourceLastAccessedTime(resourceLastAccessedTime);
	}
	
	private void keepConnectionActive(MhpUser mhpUser) {
		String sessionId = mhpUser.getMdwsInfo().getSessionId();
		ResourceLastAccessedTimeDAO dao = springContext.getBean(ResourceLastAccessedTimeDAO.class);
		//stopConnectionPolling(mhpUser.getMdwsInfo()); //Just in case we are already polling, stop it (we never should be)
		KeepAliveThread keepAliveThread = new KeepAliveThread(sessionId, mhpUser.getUserIdentifier(), this.getEmrSvcPort(), this.sessionTimeOutInSeconds, dao);
	
		final int intervalForKeepMdwsConnectionAlive = this.mdwsKeepAliveIntervalInSeconds;
	
		final ScheduledFuture<?> pollingScheduledTask = scheduler.scheduleAtFixedRate(
				keepAliveThread,
				intervalForKeepMdwsConnectionAlive,
				intervalForKeepMdwsConnectionAlive,
				SECONDS);
		scheduledTasks.put(sessionId, pollingScheduledTask);
		updateUserSessionMap(mhpUser, sessionId);
		logger.info("MdwsAppAuthenticator - Active MDWS Session");
	}
	
	private void updateUserSessionMap(MhpUser mhpUser, String sessionId) {
		List<String> sessionIds = null;
		if (null == userSessionMap.get(mhpUser.getUserIdentifier().toString())) {
			sessionIds = new ArrayList<String>();
			sessionIds.add(sessionId);
		} else {
			sessionIds = userSessionMap.get(mhpUser.getUserIdentifier().toString());
			if (!sessionIds.contains(sessionId)) {
				sessionIds.add(sessionId);
			}
		}
		userSessionMap.put(mhpUser.getUserIdentifier().toString(), sessionIds);
	}	
	
	private EmrSvcSoap getEmrSvcPort(){
		URL emrSvcWsdlUrl = this.getEmrSvcWsdlUrl();
		
		EmrSvc emrService = null;
		emrService = new EmrSvc(emrSvcWsdlUrl, new QName("http://mdws.domain/EmrSvc", "EmrSvc"));
		EmrSvcSoap port = emrService.getEmrSvcSoap();
		
		((BindingProvider)port).getRequestContext().put(BindingProvider.SESSION_MAINTAIN_PROPERTY, true);
		((BindingProvider)port).getRequestContext().put(BindingProvider.ENDPOINT_ADDRESS_PROPERTY, this.getEmrSvcUrl());	    

		return port;
	}
	
	@Override
	public void logoff(MhpUser mhpUser) {
		logger.info("MdwsAppAuthenticator - Logging Off");
		stopConnectionPolling(mhpUser.getMdwsInfo());
		disconnectMdwsConnection(mhpUser.getMdwsInfo());
	}
	
	private void disconnectMdwsConnection(MdwsInfo mdwsInfo) {
		try {
			EmrSvcSoap port = getEmrSvcPort();
			logger.info("MdwsAppAuthenticator - Disconnecting MDWS");
			MdwsAppAuthenticatorUtil.setPortSessionId(mdwsInfo.getSessionId(), port);
			port.disconnect();
			logger.info("MdwsAppAuthenticator - Disconnected port and MDWS session");
		} catch (Exception e) {
			logger.error("MdwsAppAuthenticator - Problem logging off");
		}
	}

	static void stopConnectionPolling(MdwsInfo mdwsInfo) {
		logger.info("MdwsAppAuthenticator - Removing all MDWS sessions");
		removeAllSessionsForThatUser(mdwsInfo);
	}

	private static void removeAllSessionsForThatUser(MdwsInfo mdwsInfo) {
		List<String> sessionIds  = userSessionMap.get(mdwsInfo.getUserId());
		if (null != sessionIds) {
			for (String sessionId : sessionIds) {
				try {
					ScheduledFuture<?> pollingScheduledTask = scheduledTasks.get(sessionId);
					if (pollingScheduledTask != null) {
						pollingScheduledTask.cancel(true);
						scheduledTasks.remove(sessionId);
					}
				}
				catch (Exception e) {
					logger.error("Error stopping polling mdws: " + sessionId + ": " + e.getMessage());
				}
			}
			userSessionMap.remove(mdwsInfo.getUserId());
		}
	}

}
