package com.agilex.healthcare.mobilehealthplatform.auth.provider.mdws;

import java.util.Date;

import com.agilex.healthcare.mobilehealthplatform.authorization.ResourceLastAccessedTime;
import com.agilex.healthcare.mobilehealthplatform.authorization.ResourceLastAccessedTimeDAO;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.mdws.MdwsInfo;
import com.agilex.healthcare.mobilehealthplatform.mdws.MdwsResponseException;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.EmrSvcSoap;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.UserSecurityKeyArray;
import com.agilex.healthcare.utility.DateHelper;

public class KeepAliveThread implements Runnable {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(KeepAliveThread.class);
	private String sessionId;
	private EmrSvcSoap emrPort = null;
	private int secondsToKeepPollingMdws = 15*60; //15 minutes
	private ResourceLastAccessedTimeDAO dao;
	private PatientIdentifier userIdentifier;
	

	public KeepAliveThread(String sessionId, PatientIdentifier userIdentifier, EmrSvcSoap emrPort, int secondsToKeepPollingMdws, ResourceLastAccessedTimeDAO dao) {
		this.sessionId = sessionId;
		this.userIdentifier = userIdentifier;
		this.emrPort = emrPort;
		this.secondsToKeepPollingMdws = secondsToKeepPollingMdws;
		this.dao = dao;
	}
	
	@Override
	public void run() {
		pollEmr();
	}
	
	private void pollEmr() {
		logger.debug("Keeping emrservice connection alive");

		if (isRequestWithinTimeoutBoundary(userIdentifier.toString()) == false) {
			logger.info("KeepAliveThread - Stop polling, because of MDWS session id timeout");
			stopPolling();
			return;
		}
		
		MdwsAppAuthenticatorUtil.setPortSessionId(sessionId, emrPort);
		
		try {
			UserSecurityKeyArray mdwsKeys = emrPort.getUserSecurityKeys(userIdentifier.getUniqueId());
			if (mdwsKeys.getFault() != null){
				logger.error("Error in polling for session");
				stopPolling();
			}
		}
		catch (MdwsResponseException e) {
			logger.error("Keeping emrservice connection alive mdws failure");
			stopPolling();
		}
		catch (Exception e) {
			logger.error("Keeping emrservice connection alive failure");
			stopPolling();
		}
	}

	private MdwsInfo getMdwsInfo() {
		MdwsInfo mdwsInfo = new MdwsInfo();
		mdwsInfo.setUserId(this.userIdentifier.toString());
		mdwsInfo.setSessionId(this.sessionId);
		return mdwsInfo;
	}

	private void stopPolling() {
		MdwsInfo mdwsInfo = getMdwsInfo();
		MdwsAppAuthenticator.stopConnectionPolling(mdwsInfo);
		this.emrPort.disconnect();
	}

	public String getSessionId() {
		return sessionId;
	}

	public EmrSvcSoap getEmrPort() {
		return emrPort;
	}
	
	private boolean isRequestWithinTimeoutBoundary(String userId) {
		
		if (secondsToKeepPollingMdws > DateHelper.calculateDeltaInSeconds(getLastAccessedTime(userId), new Date())) {
			return true;
		} else {
			logger.info("KeepAliveThread - Request is not within timeout boundary");
			return false;
		}
	}
	
	private Date getLastAccessedTime(String userId){
		ResourceLastAccessedTime resourceLastAccessedTime = dao.fetchResourceLastAccessedTime(userId);
		return resourceLastAccessedTime.getLastAccessedTime();
	}
	
}
