package com.agilex.healthcare.mobilehealthplatform.auth.provider;

import java.io.IOException;
import java.util.List;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.cocoon.configuration.Settings;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.session.SessionInformation;
import org.springframework.security.core.session.SessionRegistry;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;

import com.agilex.healthcare.mobilehealthplatform.domain.MhpUser;
import com.agilex.healthcare.mobilehealthplatform.oauth.StaffAppAuthenticator;
import com.agilex.healthcare.mobilehealthplatform.security.AppUser;
import com.agilex.healthcare.mobilehealthplatform.security.MhpUserFactory;
import com.agilex.healthcare.mobilehealthplatform.utils.HttpSessionCollector;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.security.oauth.TokenStoreLogout;



@Controller
public class StaffLogoutController{

	private static final Log logger = LogFactory.getLog(StaffLogoutController.class);
	
	private static final String KEY_REDIRECT_URI = "redirect_uri";

    private static final String KEY_REDIRECT_URI_ROOTS = "oauth.permitted_redirect_uri_roots";
	
	@Resource
	private TokenStoreLogout tokenStore;

	@Autowired (required=false)
	private StaffAppAuthenticator authenticator;
	
	@Autowired(required=false) 
	private SessionRegistry sessionRegistry;

    @Resource
    Settings settings;
	
	@RequestMapping(value = "/logout", method = RequestMethod.GET)
	public void logout(HttpServletRequest request, HttpServletResponse response, ModelMap model) {
		deleteToken();
		invaliateSessionAndSecurityContext(request);
		String redirectUri = determineRedirectUri(request);
		redirect(response, redirectUri);
	}

	public void deleteToken() {
		Authentication currentUser = getCurrentUser();
		if (currentUser != null && currentUser instanceof AppUser) {
			tokenStore.removeAllUSerTokens(currentUser);
		}
		
        MhpUser mhpUser = MhpUserFactory.createFromAuthentication(currentUser);
        if (mhpUser != null && mhpUser.getMdwsInfo() != null){	
        	logger.info("logging off user with MDWS session");
        	authenticator.logoff(mhpUser);
        }
	}
	
	private Authentication getCurrentUser() {
		SecurityContext ctx = SecurityContextHolder.getContext();
		return ctx.getAuthentication();
	}
	
	private void invaliateSessionAndSecurityContext(HttpServletRequest request) {
		
		Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
		
		if(authentication != null && authentication.getPrincipal() instanceof AppUser){
			List<SessionInformation> allSessions = sessionRegistry.getAllSessions(authentication.getPrincipal(), false);
			for (SessionInformation sessionInformation : allSessions) {
				HttpSession httpSession = HttpSessionCollector.find(sessionInformation.getSessionId());
				if (httpSession != null) {
					logger.info("invalidating http session Id");
					httpSession.invalidate();
				}
				sessionInformation.expireNow();
				this.sessionRegistry.removeSessionInformation(sessionInformation.getSessionId());
			}
		}
		
		SecurityContextHolder.clearContext();
	}

	private String determineRedirectUri(HttpServletRequest request) {
		
		String redirectUri = request.getParameter(KEY_REDIRECT_URI);
		
		//check if redirectUri is not null
		if (NullChecker.isNullish(redirectUri)){
			redirectUri =  "../launchpad/";
		} else if(!redirectUriIsPermitted(redirectUri)) {
			//check if redirectUri is permitted and validate it
            logger.debug("Redirect URI path is not white-listed");
			redirectUri =  "../launchpad/";
        }

		return redirectUri;
	}
	
	private void redirect(HttpServletResponse response, String redirectUri) {
        try {
			response.sendRedirect(redirectUri);
		} catch (IOException e) {
			throw new RuntimeException("Exception while redirecting");
		}
	}

    /*
    * This method ensures that the URI matches permitted base uris
    * using properties configured on the classpath.
    *
    * The property associated with KEY_REDIRECT_URI_ROOTS can be a comma
    * separated list of base URIs
    *
    * @param redirectUri
    */
    boolean redirectUriIsPermitted(String redirectUri) {
        if(NullChecker.isNullish(redirectUri))
            return false;

        String redirectUriRoots = settings.getProperty(KEY_REDIRECT_URI_ROOTS);

        if(NullChecker.isNotNullish(redirectUriRoots)) {
            String[] roots = redirectUriRoots.split(",");

            String ignoreCaseRedirectUri = redirectUri.toLowerCase();
            for (String root : roots) {
                String ignoreCaseRoot = root.toLowerCase();
                if (ignoreCaseRedirectUri.startsWith(ignoreCaseRoot))
                    return true;
            }
        }

        return false;
    }
}
