package com.agilex.healthcare.mobilehealthplatform.auth.provider;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.util.StopWatch;

import com.agilex.healthcare.mobilehealthplatform.authorization.LoginLogoutUserDAO;
import com.agilex.healthcare.mobilehealthplatform.domain.MhpUser;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.provider.auth.LoginLogoutUser;
import com.agilex.healthcare.mobilehealthplatform.provider.auth.StaffFacilityInfo;
import com.agilex.healthcare.mobilehealthplatform.security.AppAuthenticator;
import com.agilex.healthcare.mobilehealthplatform.security.AppUser;

public class StaffAuthenticationProvider implements AuthenticationProvider {

	private static final Log logger = LogFactory.getLog(StaffAuthenticationProvider.class);
	
	//This will be injected by Spring via the setter method
	AppAuthenticator appAuthenticator;
	
	@Resource
	private LoginLogoutUserDAO loginLogoutUserDAO;

	@Override
	public Authentication authenticate(Authentication inputAuthentication)
	        throws AuthenticationException {

		
		logger.debug("StaffAuthenticationProvider - attempt to Authenticate...");
		
		String username = (String) inputAuthentication.getPrincipal();
		String password = (String) inputAuthentication.getCredentials();
		String facilityName = null;
		String facilityCode = null;

		logger.info("StaffAuthenticationProvider - attempt to Authenticate User :" + username);
		
		if (inputAuthentication.getDetails() instanceof StaffFacilityInfo){
			StaffFacilityInfo details = (StaffFacilityInfo) inputAuthentication.getDetails();
			facilityCode = details.getFacilityCode();
			facilityName = details.getFacilityName();
		}

		MhpUser authenticatedUser = performAuthentication(username, password, facilityCode);
		
		if(authenticatedUser != null) {
			authenticatedUser.setVistaLocation(facilityCode);
			authenticatedUser.setFacilityName(facilityName);
		}

		Authentication outputAuthentication = buildOutputAuthenticationObject(inputAuthentication, authenticatedUser, username, password) ;
	
		updateLoginLogoutUser(authenticatedUser.getUserIdentifier());
		logger.info("StaffAuthenticationProvider - Authenticated User");
		return outputAuthentication;
	}

	private void updateLoginLogoutUser(PatientIdentifier patientIdentifier) {
		LoginLogoutUser loginLogoutUser = new LoginLogoutUser();
		loginLogoutUser.setLogout(false);
		loginLogoutUser.setUserId(patientIdentifier.toString());
		loginLogoutUser.setUpdateDateTime(new Date());
		loginLogoutUserDAO.saveLoginLogoutUser(loginLogoutUser);
	}

	private MhpUser performAuthentication(String username, String password, String vistaLocation){
		MhpUser authenticatedUser;
		//measuring elapsed time
		StopWatch stopWatch = new StopWatch("StaffAuthenticationProvider");
		stopWatch.start("PerformAuthentication");
		try {
	         authenticatedUser = appAuthenticator.authenticate(username, password, vistaLocation);
        } catch (Exception e) {
        	logger.error("auth failed");
			throw new BadCredentialsException("Access Code/Verify Code does not match");
        }
        stopWatch.stop();
        logger.info(stopWatch.prettyPrint());

		return authenticatedUser;
	}

	private Authentication buildOutputAuthenticationObject(Authentication inputAuthentication, MhpUser authenticatedUser, String username, String password) {
		
		List<GrantedAuthority> authorities = new ArrayList<GrantedAuthority>();
		
		authorities.add(new SimpleGrantedAuthority("ROLE_MHP_USER"));
		authorities.add(new SimpleGrantedAuthority("ROLE_STAFF"));

		AppUser appUser = new AppUser(username, password, authorities, authenticatedUser);
		

		return new UsernamePasswordAuthenticationToken(appUser, inputAuthentication.getCredentials(), authorities);
    }
	
	@Override
	public boolean supports(Class<? extends Object> authentication) {
		
		// Returns true if this AuthenticationProvider supports the indicated
		// Authentication object. See the javadoc on the interface for details.
		// copied it from AbstractUserDetailsAuthenticationProvider
		return (UsernamePasswordAuthenticationToken.class.isAssignableFrom(authentication));
	}

	/**
	 * setter required by IOC to inject...
	 * 
	 * @param appAuthenticator to be injected by Spring
	 */
	public void setAppAuthenticator(AppAuthenticator appAuthenticator) {
		this.appAuthenticator = appAuthenticator;
	}

}