package com.agilex.healthcare.mobilehealthplatform.security.infrastructure;

import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.util.EntityUtils;
import org.json.simple.JSONObject;
import org.json.simple.parser.*;

import java.io.IOException;

/**
 * Container for a resource response that provides a minimal interface for this lib.
 */
public class HttpClientResponse {
    private final Resource resource;
    private final int statusCode;
    private final String body;

    public HttpClientResponse(CloseableHttpResponse response) {
        assert response != null;

        statusCode = response.getStatusLine().getStatusCode();
        resource = getLocation(response);
        body = getBody(response);

        closeResponse(response);
    }

    private Resource getLocation(CloseableHttpResponse response) {
        if (response.containsHeader("Location"))
            return new Resource(response.getFirstHeader("Location").getValue());
        else
            return new Resource();
    }

    private String getBody(CloseableHttpResponse response) {
        try {
            return EntityUtils.toString(response.getEntity());
        } catch (IOException e) {
            //Ignore for now
        }

        return "";
    }

    private void closeResponse(CloseableHttpResponse response) {
        try {
            response.close();
        } catch (IOException e) {
            throw new RuntimeException("Unable to close http response.", e);
        }
    }

    public Value getCode() {
        return resource.getCode();
    }

    public Resource getResource() {
        return resource;
    }

    public Value getToken() {
        if (statusCode == 401)
            throw new RuntimeException("Invalid token request.  Possible cause is 'Bad client credentials'.");

        if (statusCode == 400)
            throw new RuntimeException("Invalid token request. Possible cause is 'Redirect URI mismatch.");

        if ("".equals(body))
            return new Value();

        String token;

        JSONParser jsonParser = new JSONParser();
        try {
            JSONObject jsonObject = (JSONObject) jsonParser.parse(body);
            token = (String) jsonObject.get("access_token");
        } catch (ParseException e) {
            throw new RuntimeException("Unable to parse body to get access token.", e);
        }

        return new Value(token);
    }
}
