package com.agilex.healthcare.mobilehealthplatform.security;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import javax.annotation.Resource;
import javax.ws.rs.core.MediaType;

import org.junit.Test;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.security.oauth2.common.OAuth2AccessToken;

import com.agilex.healthcare.mobilehealthplatform.dto.HAOauth2Authentication;
import com.agilex.healthcare.mobilehealthplatform.dto.TokenValidationRequest;
import com.agilex.healthcare.mobilehealthplatform.oauth.OauthClientDetails;
import com.agilex.healthcare.mobilehealthplatform.oauth.OauthJdbcClientDetailsService;
import com.agilex.healthcare.mobilehealthplatform.security.infrastructure.HttpClient;
import com.agilex.healthcare.mobilehealthplatform.security.infrastructure.Value;
import com.sun.jersey.api.client.UniformInterfaceException;

/**
 * Add a Description.  Should include class purpose (why not what) and assumptions.
 */
public class StaffClientTests extends AuthSpringContextConfig {

    private static final String HOST_URI = "http://localhost:8080";
    private static final String REDIRECT_URI = "https://localhost:8080/launchpad/";
    private static final String STAFF_AUTHORIZE_BASE_URI = "/AuthorizationServices/provider";
    private static final String tokenValidationUri = "http://localhost:8080/AuthorizationServices/rest/validateToken";
    private static final String MHP_BASE_URL = "http://localhost:8080/MobileHealthPlatformWeb";
            
    @Resource
    private OauthJdbcClientDetailsService oauthJdbcClientDetailsService;
    
    @Test
    public void getAccessTokenForStaff() {
        //MBBWEB client Login 
        Client client = getClient("MBBWEB", "MobileBlueButton");
        Value token = getAccessToken("MBBWEB", "MobileBlueButton");
        User staff = getUser("MBBWEB", "MobileBlueButton");

        //Use Token in Authorization header on calls to your REST resources
        //Key: Authorization , Value: Bearer e89e3ddd-7c9d-48aa-9ed3-27420b964894
 
        System.out.println("Token: " + token.toString());
        assertTrue(token.toString().length() > 0);

        client.logoutAndCloseConnection(staff, token);
    }

    @Test
    public void getAccessTokenForInvalidStaff() {
        HttpClient httpClient = new HttpClient(HOST_URI);
        Client client = getClient("MBBWEB", "MobileBlueButton");
        
        User staff = new User(httpClient, "invalid_patient", "cprs4321$", "777", "TEST VAMC 3", STAFF_AUTHORIZE_BASE_URI);

        try {
            Value code = client.authorize(staff);
        } catch (RuntimeException r) {
            assertEquals("Unable to log in user.  Make sure user credentials are correct.", r.getMessage());
        }
    }

    @Test
    public void validAccessTokeDoesNotAllowAccessToData() {
        Client client = getClient("MBBWEB", "MobileBlueButton");
        Value token = getAccessToken("MBBWEB", "MobileBlueButton");
        User staff = getUser("MBBWEB", "MobileBlueButton");
        System.out.println("Token: " + token.toString());
        assertTrue(token.toString().length() > 0);

        token = new Value(token.toString());
        assertDataRetrieve(client, token, HttpStatus.OK, MHP_BASE_URL + "/rest/patient/EDIPI/D123401");
        client.logoutAndCloseConnection(staff, token);
    }
    
    @Test
    public void invalidAccessTokeDoesNotAllowAccessToData() {
        Client client = getClient("MBBWEB", "MobileBlueButton");
        Value token = getAccessToken("MBBWEB", "MobileBlueButton");
        User staff = getUser("MBBWEB", "MobileBlueButton");
        System.out.println("Token: " + token.toString());
        assertTrue(token.toString().length() > 0);

        token = new Value(token.toString() + "X");
        assertDataRetrieve(client, token, HttpStatus.UNAUTHORIZED, MHP_BASE_URL + "/rest/patient/EDIPI/D123401");
        client.logoutAndCloseConnection(staff, token);
    }
    
    @Test
    public void authenticateWithAuthorizationCode() throws Exception {
            validateAuthenticationWithOauth("MobileBlueButton", "MBBWEB");
    }
    
    @Test
    public void getAccessTokensForMultipleClientsAndLogoutP2P() {
        //MBBWEB client Login 
        Client client = getClient("MBBWEB", "MobileBlueButton");
        Value token = getAccessToken("MBBWEB", "MobileBlueButton");
        User staff = getUser("MBBWEB", "MobileBlueButton");
        System.out.println("Token: " + token.toString());
        assertTrue(token.toString().length() > 0);

        
        //P2P client Login
        Value token1 = getAccessToken("P2P", "p2p");
        User staff1 = getUser("P2P", "p2p");
        System.out.println("Token1: " + token1.toString());
        assertTrue(token1.toString().length() > 0);
        
        //P2P client logout
        client.logoutAndCloseConnection(staff1, token1);
        
        //Assert if MBBWEB client token is still active
        TokenValidationRequest tokenValidationRequest = new TokenValidationRequest();
        tokenValidationRequest.setToken(token.toString());
        
        HAOauth2Authentication haOauth2Authentication = validateToken(tokenValidationRequest);
        assertEquals("vehu010", haOauth2Authentication.getUserName());
        assertEquals("MobileBlueButton", haOauth2Authentication.getAuthorizatioRequest().getClientId());
        assertEquals("20012", haOauth2Authentication.getMhpUser().getId());
        assertTrue(haOauth2Authentication.getAuthorities().toString().contains("ROLE_STAFF"));
        assertNotNull(haOauth2Authentication.getLastLoginDate());
        
        assertTrue(token.toString().length() > 0);
        //MBBWEB Client Logout
        client = getClient("MBBWEB", "MobileBlueButton");
        token = getAccessToken("MBBWEB", "MobileBlueButton");
        System.out.println("Token: " + token.toString());
        staff = getUser("MBBWEB", "MobileBlueButton");
        client.logoutAndCloseConnection(staff, token);
    }

    @Test
    public void getAccessTokensForMultipleClientsAndLogoutMBBWEB() {
        //MBBWEB client Login 
        Client client = getClient("MBBWEB", "MobileBlueButton");
        Value token = getAccessToken("MBBWEB", "MobileBlueButton");
        User staff = getUser("MBBWEB", "MobileBlueButton");
        System.out.println("Token: " + token.toString());
        assertTrue(token.toString().length() > 0);
        
        //P2P client Login
        Value token1 = getAccessToken("P2P", "p2p");
        User staff1 = getUser("P2P", "p2p");
        System.out.println("Token1: " + token1.toString());
        assertTrue(token1.toString().length() > 0);
        
        //MBBWEB client logout
        client.logoutAndCloseConnection(staff, token);
        
        //Assert if P2P Client token is still active
        TokenValidationRequest tokenValidationRequest = new TokenValidationRequest();
        tokenValidationRequest.setToken(token1.toString());
        
        HAOauth2Authentication haOauth2Authentication = validateToken(tokenValidationRequest);
        assertEquals("vehu010", haOauth2Authentication.getUserName());
        assertEquals("p2p", haOauth2Authentication.getAuthorizatioRequest().getClientId());
        assertEquals("20012", haOauth2Authentication.getMhpUser().getId());
        assertTrue(haOauth2Authentication.getAuthorities().toString().contains("ROLE_STAFF"));
        assertNotNull(haOauth2Authentication.getLastLoginDate());
        
        assertTrue(token1.toString().length() > 0);
        //P2P Client Logout
        client = getClient("P2P", "p2p");
        token1 = getAccessToken("P2P", "p2p");
        staff1 = getUser("P2P", "p2p");
        System.out.println("Token1: " + token1.toString());
        client.logoutAndCloseConnection(staff1, token1);
    }

    @Test
    public void getAccessTokenForStaffWithClientP2PTimeout() throws Exception {
        
        OauthClientDetails clientDetails = (OauthClientDetails) oauthJdbcClientDetailsService.loadClientByClientId("p2p");
        //Update the timeout value to 1 min.
        oauthJdbcClientDetailsService.updateClientTimeout("p2p", "60");

        //P2P client Login 
        Client client = getClient("P2P", "p2p");
        Value token = getAccessToken("P2P", "p2p");
        User staff = getUser("P2P", "p2p");
        System.out.println("Token: " + token.toString());
        assertTrue(token.toString().length() > 0);
        
        TokenValidationRequest tokenValidationRequest = new TokenValidationRequest();
        tokenValidationRequest.setToken(token.toString());
        
        Thread.sleep(60003);
        
        HAOauth2Authentication haOauth2Authentication = null;
        try {
            haOauth2Authentication = validateToken(tokenValidationRequest);
        } catch (UniformInterfaceException e) {
           //Returns Status code - 204
           assertTrue(e.getMessage().contains("204"));
        }
        
        assertTrue(token.toString().length() > 0);
        
        //Reset the timeout value.
        oauthJdbcClientDetailsService.updateClientTimeout("p2p", String.valueOf(clientDetails.getTimeoutSeconds()));
        client.logoutAndCloseConnection(staff, token);
    }

    @Test
    public void getAccessTokenForStaffWithClientP2P() throws Exception {
        
        //P2P client Login 
        Client client = getClient("P2P", "p2p");
        Value token = getAccessToken("P2P", "p2p");
        User staff = getUser("P2P", "p2p");
        System.out.println("Token: " + token.toString());
        assertTrue(token.toString().length() > 0);
        
        TokenValidationRequest tokenValidationRequest = new TokenValidationRequest();
        tokenValidationRequest.setToken(token.toString());
        Thread.sleep(20001);
        HAOauth2Authentication haOauth2Authentication = null;
        haOauth2Authentication = validateToken(tokenValidationRequest);
        assertEquals("vehu010", haOauth2Authentication.getUserName());
        assertEquals("p2p", haOauth2Authentication.getAuthorizatioRequest().getClientId());
        assertEquals("20012", haOauth2Authentication.getMhpUser().getId());
        assertTrue(haOauth2Authentication.getAuthorities().toString().contains("ROLE_STAFF"));
        assertNotNull(haOauth2Authentication.getLastLoginDate());
        
        assertTrue(token.toString().length() > 0);
        client.logoutAndCloseConnection(staff, token);
    }
    
    private void validateAuthenticationWithOauth(String clientId, String clientPassword) throws Exception {
        Client client = getClient("MBBWEB", "MobileBlueButton");
        Value token = getAccessToken("MBBWEB", "MobileBlueButton");
        User staff = getUser("MBBWEB", "MobileBlueButton");
        assertDataRetrieve(client, token, HttpStatus.OK, MHP_BASE_URL + "/rest/patient/EDIPI/D123401");
        client.logoutAndCloseConnection(staff, token);
    }

    
    private void assertDataRetrieve(Client client, Value accessToken, HttpStatus expectedHttpStatus, String serviceUri) {
        HttpHeaders headers = new HttpHeaders();
        headers.set("Authorization", String.format("%s %s", OAuth2AccessToken.BEARER_TYPE, accessToken.toString()));
        assertEquals(expectedHttpStatus, client.getStatusCode(serviceUri, headers));
    }
    
    private HAOauth2Authentication validateToken(TokenValidationRequest tokenValidationRequest) {
        com.sun.jersey.api.client.Client restClient = new com.sun.jersey.api.client.Client();
        HAOauth2Authentication haOauth2Authentication = restClient.resource(tokenValidationUri).accept(MediaType.APPLICATION_XML).post(HAOauth2Authentication.class,tokenValidationRequest);
        return haOauth2Authentication;
    }
    
    private User getUser(String clientSecret, String clientId) {
        HttpClient httpClient = new HttpClient(HOST_URI);
        User staff = new User(httpClient, "vehu010", "ve010hu!", "688", "WASHINGTON", STAFF_AUTHORIZE_BASE_URI);
        return staff;
    }
    
    private Client getClient(String clientSecret, String clientId) {
        HttpClient httpClient = new HttpClient(HOST_URI);

        ClientIdentification clientIdentification = new ClientIdentification(clientSecret, clientId, REDIRECT_URI);
        Client client = Client.createClient(httpClient, clientIdentification, STAFF_AUTHORIZE_BASE_URI);
        return client;
    }

    private Value getAccessToken(String clientSecret, String clientId) {
        HttpClient httpClient = new HttpClient(HOST_URI);

        ClientIdentification clientIdentification = new ClientIdentification(clientSecret, clientId, REDIRECT_URI);
        Client client = Client.createClient(httpClient, clientIdentification, STAFF_AUTHORIZE_BASE_URI);
        User staff = new User(httpClient, "vehu010", "ve010hu!", "688", "WASHINGTON", STAFF_AUTHORIZE_BASE_URI);

        Value code = client.authorize(staff);
        Value token = client.getAccessToken(code);
        return token;
    }

    
    
}
