package com.agilex.healthcare.mobilehealthplatform.security;

import java.io.IOException;

import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.http.client.ClientHttpRequest;
import org.springframework.http.client.ClientHttpResponse;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.web.client.RequestCallback;
import org.springframework.web.client.ResponseErrorHandler;
import org.springframework.web.client.ResponseExtractor;
import org.springframework.web.client.RestTemplate;

import com.agilex.healthcare.mobilehealthplatform.security.infrastructure.HttpClient;
import com.agilex.healthcare.mobilehealthplatform.security.infrastructure.HttpClientResponse;
import com.agilex.healthcare.mobilehealthplatform.security.infrastructure.Resource;
import com.agilex.healthcare.mobilehealthplatform.security.infrastructure.Value;

/**
 * Represents a client application that must interact with the Authorization service and possible a 3rd party log in
 * service for user authentication.
 *
 * A client would these things to get an access token before making any call to your resources.  Your resources should
 * require an access token which is validated before allow the resource method to be executed.
 */
public class Client {
    //The root domain should be initialized in the HttpClient instance (http://localhost:8080).
    private static final String AUTHORIZE_URI = "/authorize";
    private static final String ACCESS_TOKEN_URI = "/AuthorizationServices/oauth/token";
    private static final String DELETE_TOKEN_URI = "/AuthorizationServices/rest/token";

    private final HttpClient httpClient;
    private final String authorizeUri;
    private final ClientIdentification clientIdentification;

    public static Client createClient(HttpClient httpClient, ClientIdentification clientIdentification, String baseUri) {
        return new Client(httpClient, clientIdentification, baseUri + AUTHORIZE_URI);
    }

    private Client(HttpClient httpClient, ClientIdentification clientIdentification, String authorizeUri) {
        this.httpClient = httpClient;
        this.clientIdentification = clientIdentification;
        this.authorizeUri = authorizeUri;
    }

    public Value authorize(User user) {
        UserAgent userAgent = new UserAgent(httpClient);
        Resource resource = clientIdentification.buildAuthorizationResource(authorizeUri);

        return userAgent.authorize(resource, user);
    }

    public Value getAccessToken(Value code) {
        Resource resource = new Resource(ACCESS_TOKEN_URI);

        HttpClientResponse response = resource.post(httpClient, clientIdentification.getAccessTokenParameters(code.toString()));
        Value token = response.getToken();

        if (!token.isSet())
            throw new RuntimeException("Access token not found.");

        return token;
    }

    //This should be called after tests are complete to free resources and prevent memory leaks.
    public void logoutAndCloseConnection(User user, Value token) {
        Resource deleteResource = new Resource(DELETE_TOKEN_URI);

        deleteResource.delete(httpClient, token);
        user.logout();
        httpClient.close();
    }
    
    public RestTemplate getRestTemplate() {
        RestTemplate client = new RestTemplate();
        HttpComponentsClientHttpRequestFactory requestFactory = new HttpComponentsClientHttpRequestFactory();
        client.setRequestFactory(requestFactory);
        client.setErrorHandler(new ResponseErrorHandler() {
                public boolean hasError(ClientHttpResponse response) throws IOException {
                        return false;
                }

                public void handleError(ClientHttpResponse response) throws IOException {
                }
        });
        return client;
    }
    
    public HttpStatus getStatusCode(String path, final HttpHeaders headers) {
        RequestCallback requestCallback = createRequestCallback(headers);
        
        return execute(path, HttpMethod.GET, requestCallback);
    }
    
    private RequestCallback createRequestCallback(final HttpHeaders headers) {
        RequestCallback requestCallback = new NullRequestCallback();
        if (headers != null) {
                requestCallback = new RequestCallback() {
                        public void doWithRequest(ClientHttpRequest request) throws IOException {
                                request.getHeaders().putAll(headers);
                        }
                };
        }
        return requestCallback;
    }
    
    private HttpStatus execute(String path, HttpMethod method, RequestCallback requestCallback) {
        HttpStatus statusCode = getRestTemplate().execute(path, method, requestCallback, new ResponseExtractor<ResponseEntity<String>>() {
                public ResponseEntity<String> extractData(ClientHttpResponse response) throws IOException {
                        return new ResponseEntity<String>(response.getStatusCode());
                }
        }).getStatusCode();
        
        return statusCode;
    }
    
    private static final class NullRequestCallback implements RequestCallback {
        public void doWithRequest(ClientHttpRequest request) throws IOException {
        }
    }    
    
}
