/**
 * 
  Package: MAG - VistA Imaging
  WARNING: Per VHA Directive 2004-038, this routine should not be modified.
  Date Created: Jun 23, 2009
  Site Name:  Washington OI Field Office, Silver Spring, MD
  Developer:  vhaiswwerfej
  Description: 

        ;; +--------------------------------------------------------------------+
        ;; Property of the US Government.
        ;; No permission to copy or redistribute this software is given.
        ;; Use of unreleased versions of this software requires the user
        ;;  to execute a written test agreement with the VistA Imaging
        ;;  Development Office of the Department of Veterans Affairs,
        ;;  telephone (301) 734-0100.
        ;;
        ;; The Food and Drug Administration classifies this software as
        ;; a Class II medical device.  As such, it may not be changed
        ;; in any way.  Modifications to this software may result in an
        ;; adulterated medical device under 21CFR820, the use of which
        ;; is considered to be a violation of US Federal Statutes.
        ;; +--------------------------------------------------------------------+

 */
package gov.va.med.imaging.ihe.xca.proxy;

import gov.va.med.imaging.artifactsource.ResolvedArtifactSource;
import gov.va.med.imaging.core.interfaces.exceptions.ConnectionException;
import gov.va.med.imaging.core.interfaces.exceptions.MethodException;
import gov.va.med.imaging.exchange.business.DocumentFilter;
import gov.va.med.imaging.exchange.business.ImageStreamResponse;
import gov.va.med.imaging.exchange.business.Requestor;
import gov.va.med.imaging.exchange.business.documents.DocumentSetResult;
import gov.va.med.imaging.ihe.XCATranslatorAdb;
import gov.va.med.imaging.ihe.XCATranslatorXmlBeans;
import gov.va.med.imaging.ihe.exceptions.TranslationException;
import gov.va.med.imaging.ihe.xca.datasource.XCADocumentDataSourceService;
import gov.va.med.imaging.ihe.xca.datasource.configuration.XCADataSourceConfiguration;
import gov.va.med.imaging.proxy.ssl.AuthSSLProtocolSocketFactory;
import gov.va.med.imaging.transactioncontext.TransactionContext;
import gov.va.med.imaging.transactioncontext.TransactionContextFactory;
import gov.va.med.imaging.transactioncontext.TransactionContextHttpHeaders;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLDecoder;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.apache.axis2.AxisFault;
import org.apache.axis2.addressing.EndpointReference;
import org.apache.axis2.client.Options;
import org.apache.axis2.client.ServiceClient;
import org.apache.axis2.context.ConfigurationContext;
import org.apache.axis2.context.ConfigurationContextFactory;
import org.apache.axis2.context.MessageContext;
import org.apache.axis2.transport.http.HTTPConstants;
import org.apache.axis2.transport.http.HttpTransportProperties;
import org.apache.commons.httpclient.protocol.DefaultProtocolSocketFactory;
import org.apache.commons.httpclient.protocol.Protocol;
import org.apache.commons.httpclient.protocol.ProtocolSocketFactory;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

/**
 * @author vhaiswwerfej
 *
 */
public class XCADataSourceProxy 
{
	// The "protocol" to use for secure connections set up using the truststore
	// and keystore specified in the configuration
	public static final String SECURE_XCA_VIRTUAL_PROTOCOL = "xcas";
	public static final String XCA_VIRTUAL_PROTOCOL = "xca";
	
	/**
	 * 
	 */
	//private static final String DEFAULT_HOME_COMMUNITY_OID = "urn:oid:2.16.840.1.113883.3.166";
	private final static Logger logger = LogManager.getLogger(XCADataSourceProxy.class);
	private static ConfigurationContext configContext;

	static
	{
        String home = System.getenv("CATALINA_HOME");
        home = home == null ? System.getProperty("catalina.home") : home;
        
        // create this folder at your home. This folder could be anything
        //then create the "modules" folder

        File repository = new File(home + File.separator + "axis2-repository");
        if (!repository.exists()) 
            LogManager.getLogger(XCADataSourceProxy.class).error("AXIS2 repository '" + repository.getAbsolutePath() + "' does not exist");
        
        //copy the LoggingModule.mar to "modules" folder.
        //then modify the axis2.xml that is generating there according to
        //phases that being included in the "module.xml"
        try
		{
			configContext = ConfigurationContextFactory.
				createConfigurationContextFromFileSystem(repository.getAbsolutePath(), repository.getAbsolutePath() + "/conf/client-axis2.xml");
			ServiceClient serviceClient = new ServiceClient(configContext, null);
			serviceClient.engageModule("addressing");
		}
		catch (AxisFault x)
		{
			x.printStackTrace();
		}
	}
	
	// ==========================================================================================
	// 
	// ==========================================================================================
	
	private final XCATranslatorXmlBeans xmlBeansTranslator;
	private final XCATranslatorAdb adbTranslator;
	private final gov.va.med.imaging.ihe.xca.xmlbeans.RespondingGateway_ServiceStub queryStub;
	private final gov.va.med.imaging.ihe.xca.adb.RespondingGatewayRetrieve_ServiceStub retrieveStub;
	private final ResolvedArtifactSource resolvedArtifactSource;
	private final URL queryUrl;
	private final URL retrieveUrl;
	private final XCADataSourceConfiguration configuration;
	
	/**
	 * The ResolvedArtifactSource passed to this constructor must have only one
	 * URL for the query and one for the retrieve.  Any URL fixup in protocol
	 * or path must be completed before this constructor.  This instance will use
	 * the first URL in each list without modification.
	 * 
	 * @param siteConfiguration
	 * @param site
	 * @param homeCommunityOid
	 * @throws ConnectionException
	 */
	public XCADataSourceProxy(
		ResolvedArtifactSource resolvedArtifactSource, 
		XCADataSourceConfiguration configuration)
	throws ConnectionException
	{
		this.configuration = configuration;
		this.xmlBeansTranslator = new XCATranslatorXmlBeans();
		this.adbTranslator = new XCATranslatorAdb();

		this.resolvedArtifactSource = resolvedArtifactSource;
		
		List<String> queryAuthSchemeList = new ArrayList<String>();
		queryAuthSchemeList.add(HttpTransportProperties.Authenticator.BASIC);
		//queryAuthSchemeList.add(HttpTransportProperties.Authenticator.DIGEST);
		//queryAuthSchemeList.add(HttpTransportProperties.Authenticator.NTLM);
		
		List<String> retrieveAuthSchemeList = new ArrayList<String>();
		retrieveAuthSchemeList.add(HttpTransportProperties.Authenticator.BASIC);
		//retrieveAuthSchemeList.add(HttpTransportProperties.Authenticator.DIGEST);
		//retrieveAuthSchemeList.add(HttpTransportProperties.Authenticator.NTLM);

		if(resolvedArtifactSource.getMetadataUrls().size() < 1)
			throw new ConnectionException("The resolved artifact source '" + resolvedArtifactSource.toString() +"' does not support the required protocol '" + XCADocumentDataSourceService.SUPPORTED_PROTOCOL + "' for metadata.");
		if(resolvedArtifactSource.getArtifactUrls().size() < 1)
			throw new ConnectionException("The resolved artifact source '" + resolvedArtifactSource.toString() +"' does not support the required protocol '" + XCADocumentDataSourceService.SUPPORTED_PROTOCOL + "' for artifacts.");
		
		// set this before creating the stubs, we use getters that depend
		// on these values in the stub creation
		this.queryUrl = resolvedArtifactSource.getMetadataUrls().get(0);
		this.retrieveUrl = resolvedArtifactSource.getArtifactUrls().get(0);
		
		// requires that the query and retrieve URL fields be set
		validateProtocolRegistration();

		try
		{
			//queryUrl = new URL("http://IP              /DocSearchXcaService/RespondingGateway_Service");
			//queryUrl = new URL("http://localhost:PORT/DocSearchXcaService/RespondingGateway_Service");
			this.queryStub = new gov.va.med.imaging.ihe.xca.xmlbeans.RespondingGateway_ServiceStub(configContext, this.getQueryUrlWithoutUserInfo());
			if(getQueryUrlUserID() != null)
			{
				HttpTransportProperties.Authenticator queryAuthorization = new HttpTransportProperties.Authenticator();
				queryAuthorization.setUsername(getQueryUrlUserID());
				if(getQueryUrlPassword() != null)
					queryAuthorization.setPassword(getQueryUrlPassword());
				queryAuthorization.setAuthSchemes(queryAuthSchemeList);
				queryAuthorization.setPreemptiveAuthentication(true);
				Options queryOptions = 
					queryStub._getServiceClient().getOptions() != null ? 
						queryStub._getServiceClient().getOptions() :
						new Options();
				queryOptions.setProperty(HTTPConstants.AUTHENTICATE, queryAuthorization);
				queryStub._getServiceClient().setOptions(queryOptions);
			}
			ServiceClient queryServiceClient = this.queryStub._getServiceClient();
			System.out.println( "Query service client is of type '" + queryServiceClient.getClass().getCanonicalName() + "'." );
			
			this.retrieveStub = new gov.va.med.imaging.ihe.xca.adb.RespondingGatewayRetrieve_ServiceStub(configContext, this.getRetrieveUrlWithoutUserInfo());
			Options retrieveOptions = 
				retrieveStub._getServiceClient().getOptions() != null ? 
					retrieveStub._getServiceClient().getOptions() :  
					new Options();
			if(getRetrieveUrlUserID() != null)
			{
				HttpTransportProperties.Authenticator retrieveAuthorization = new HttpTransportProperties.Authenticator();
				retrieveAuthorization.setUsername(getRetrieveUrlUserID());
				if(getRetrieveUrlPassword() != null)
					retrieveAuthorization.setPassword(getRetrieveUrlPassword());
				retrieveAuthorization.setAuthSchemes(retrieveAuthSchemeList);
				retrieveOptions.setProperty(HTTPConstants.AUTHENTICATE, retrieveAuthorization);
			}
			//retrieveOptions.setProperty(MessageContextConstants.TRANSPORT_URL,"**https://myservices.test.com/testharness/myapplicationapi.asmx");*
			// put the real protocol into the WS-Addressing "to" element
			URL httpRetrieveUrl = getRetrieveUrl();
			if("xca".equals(getRetrieveUrl().getProtocol()) )
				httpRetrieveUrl = new URL("http", getRetrieveUrl().getHost(), getRetrieveUrl().getPort(), getRetrieveUrl().getFile());
			else if("xcas".equals(getRetrieveUrl().getProtocol()) )
				httpRetrieveUrl = new URL("https", getRetrieveUrl().getHost(), getRetrieveUrl().getPort(), getRetrieveUrl().getFile());
			logger.info("Setting WS-Addressing 'to' to endpoint reference '" + httpRetrieveUrl.toString() + "'.");
			retrieveOptions.setTo(new EndpointReference(httpRetrieveUrl.toString())); 		
			retrieveOptions.setTimeOutInMilliSeconds(getRetrieveTimeout());			
			
			retrieveStub._getServiceClient().setOptions(retrieveOptions);
			
			//ServiceClient retrieveServiceClient = this.retrieveStub._getServiceClient();
			//System.out.println( "Retrieve service client is of type '" + retrieveServiceClient.getClass().getCanonicalName() + "'." );
		}
		catch(AxisFault afX)
		{
			logger.error("Error creating XCA proxy", afX);
			throw new ConnectionException(afX);
		}
		catch (MalformedURLException murlX)
		{
			logger.error("Error creating XCA proxy", murlX);
			throw new ConnectionException(murlX);
		}
	}
	
	private int getQueryTimeout()
	{
		if(configuration.getQueryTimeout() == null)
			return XCADataSourceConfiguration.defaultQueryTimeout;
		return configuration.getQueryTimeout();
	}
	
	private int getRetrieveTimeout()
	{
		if(configuration.getRetrieveTimeout() == null)
			return XCADataSourceConfiguration.defaultRetrieveTimeout;
		return configuration.getRetrieveTimeout();
	}
	
	/**
	 * @param queryUrl2
	 * @param retrieveUrl2
	 */
	private void validateProtocolRegistration()
	{
		// this is synchronized with the class instance so that multiple instances
		// do not try to register the protocol socket factory
		synchronized(XCADataSourceProxy.class)
		{
			if(! isProtocolHandlerRegistered(getQueryUrl().getProtocol()))
				registerProtocolHandler(getQueryUrl().getProtocol());
			if(! isProtocolHandlerRegistered(getRetrieveUrl().getProtocol()))
				registerProtocolHandler(getRetrieveUrl().getProtocol());
		}
	}

	/**
	 * @param protocol
	 */
	private void registerProtocolHandler(String protocol)
	{
		if( XCA_VIRTUAL_PROTOCOL.equalsIgnoreCase(protocol) )
		{
			Protocol httpProtocol = Protocol.getProtocol("http");
			if(httpProtocol == null)
			{
				logger.error("Failed to register protocol '" + protocol + "' because mapped http protocol is not registered, attempting to create a default socket factory.");
				Protocol.registerProtocol( XCA_VIRTUAL_PROTOCOL, new Protocol(protocol, new DefaultProtocolSocketFactory(), 80) );
				logger.info("Protocol '" + protocol + "' registered to a new default socket factory.");
			}
			else
				Protocol.registerProtocol(XCA_VIRTUAL_PROTOCOL, httpProtocol );
		}
		else if( SECURE_XCA_VIRTUAL_PROTOCOL.equalsIgnoreCase(protocol) )
		{
			try
			{
				// Either keystore or truststore may be null but not both
				// or AuthSSLProtocolSocketFactory will fail to construct.
				URL keystoreUrl = new URL(configuration.getKeystoreUrl());	// the keystore containing the key to send as the client
				URL truststoreUrl = new URL(configuration.getTruststoreUrl());	// the keystore containing the trusted certificates, to validate the server cert against
						    		   
				ProtocolSocketFactory socketFactory = 
				    new AuthSSLProtocolSocketFactory(
				    	keystoreUrl, configuration.getKeystorePassword(), 
				    	truststoreUrl, configuration.getTruststorePassword());
				Protocol httpsProtocol = new Protocol(protocol, socketFactory, configuration.getTLSPort());

				// register our socket factory using the 'virtual' scheme "xcas".
				// When creating a socket connection use "xcas" in the URI e.g. 
				// HttpClient httpclient = new HttpClient();
				// GetMethod httpget = new GetMethod("xcas://www.whatever.com/");
				Protocol.registerProtocol(SECURE_XCA_VIRTUAL_PROTOCOL, httpsProtocol);
			}
			catch (MalformedURLException x)
			{
				x.printStackTrace();
				logger.error("Failed to register protocol '" + protocol + "' unable to form valid trust or key store URL [" + x.getMessage() + "].");
			}
		}
		else
			logger.error("Protocol '" + protocol + "' is not registered with a socket factory.");
		
		if(isProtocolHandlerRegistered(protocol))
			logger.info("Protocol '" + protocol + "' registered successfully.");
		else
			logger.info("Protocol '" + protocol + "' failed to register.");
	}

	/**
	 * @param protocol
	 * @return
	 */
	private boolean isProtocolHandlerRegistered(String protocol)
	{
		try
		{
			return Protocol.getProtocol(protocol) != null;
		}
		catch (IllegalStateException x)
		{
			return false;
		}
	}

	/**
	 * @return the queryUrl as passed to the constructor for this class
	 */
	public URL getQueryUrl()
	{
		return this.queryUrl;
	}
	
	/**
	 * @return the retrieveUrl as passed to the constructor for this class
	 */
	public URL getRetrieveUrl()
	{
		return this.retrieveUrl;
	}

	private String getQueryUrlWithoutUserInfo()
	{
		return getUrlWithoutUserInfo(getQueryUrl());
	}

	private String getRetrieveUrlWithoutUserInfo()
	{
		return getUrlWithoutUserInfo(getRetrieveUrl());
	}
	
	private String getUrlWithoutUserInfo(URL url)
	{
		return url.getProtocol() + 
			"://" +
			url.getHost() +
			(url.getPort() > 0 ? ":" + url.getPort() : "") +
			url.getFile();
	}

	private String getQueryUrlUserID()
	{
		return parseUserID(getQueryUrl().getUserInfo());
	}
	private String getQueryUrlPassword()
	{
		return parsePassword(getQueryUrl().getUserInfo());
	}
	
	private String getRetrieveUrlUserID()
	{
		return parseUserID(getRetrieveUrl().getUserInfo());
	}
	private String getRetrieveUrlPassword()
	{
		return parsePassword(getRetrieveUrl().getUserInfo());
	}

	private final static String utf8 = "UTF-8"; 
	private String parseUserID(String userInfo)
	{
		if(userInfo != null)
		{
			String[] userInfoComponents = userInfo.split(":");
			if(userInfoComponents[0] != null)
			{
				try
				{
					return URLDecoder.decode(userInfoComponents[0], utf8);
				}
				catch (UnsupportedEncodingException e)
				{
					//e.printStackTrace();
					logger.warn("Error URL decoding username '" + userInfoComponents[0] + "'.");
					return userInfoComponents[0];
				}
			}
		}
		return null;
	}
	private String parsePassword(String userInfo)
	{
		if(userInfo != null)
		{
			String[] userInfoComponents = userInfo.split(":");
			if(userInfoComponents.length > 1)
			{
				String pwd = userInfoComponents[1];
				if(pwd != null)
				{
					try
					{
						return URLDecoder.decode(pwd, utf8);
					}
					catch (UnsupportedEncodingException e)
					{
						//e.printStackTrace();
						logger.warn("Error URL decoding password '" + pwd + "'.");
						return pwd;
					}
				}
			}
		}
		return null;
	}

	/**
	 * @return the configuration
	 */
	public XCADataSourceConfiguration getConfiguration()
	{
		return this.configuration;
	}

	public XCATranslatorXmlBeans getXmlBeansTranslator()
	{
		return this.xmlBeansTranslator;
	}

	public XCATranslatorAdb getAdbTranslator()
	{
		return this.adbTranslator;
	}
	
	/**
	 * 
	 * @param patientIcn
	 * @param filter
	 * @return
	 * @throws MethodException
	 * @throws ConnectionException
	 */
	public DocumentSetResult getPatientDocumentSets(
		DocumentFilter filter)	
	throws MethodException, ConnectionException
	{
		//StoredQueryParameterSet requestParameters = new StoredQueryParameterSet();
		//CrossGatewayQueryRequest request = new CrossGatewayQueryRequest(
		//	getHomeCommunityOid(),
		//	-1L,
		//	true,
		//	requestParameters
		//);
		
		validateUrl(getQueryUrl());
		
		org.oasis.ebxml.regrep.query.xmlbeans.AdhocQueryRequestDocument request;
		try
		{
			getXmlBeansTranslator();
			request = XCATranslatorXmlBeans.translate(filter);
		}
		catch (MalformedURLException x)
		{
			logger.error("Error creating a query instance", x);
			throw new ConnectionException(x);
		}
		
		try
		{
			TransactionContext transactionContext = TransactionContextFactory.get();
			transactionContext.addDebugInformation("Executing CrossGatewayQuery to url '" + getQueryUrlWithoutUserInfo() + "'");
			logger.info("Executing CrossGatewayQuery to url '" + getQueryUrlWithoutUserInfo() + "'");
			
			// force SOAP 1.1
			//queryStub._getServiceClient().getOptions().setSoapVersionURI(org.apache.axiom.soap.SOAP11Constants.SOAP_ENVELOPE_NAMESPACE_URI);
			// force SOAP 1.2
			queryStub._getServiceClient().getOptions().setSoapVersionURI(org.apache.axiom.soap.SOAP12Constants.SOAP_ENVELOPE_NAMESPACE_URI);
			// set SOAP action
			queryStub._getServiceClient().getOptions().setProperty(org.apache.axis2.transport.http.HTTPConstants.HEADER_SOAP_ACTION, "urn:ihe:iti:2007:CrossGatewayQuery");
			queryStub._getServiceClient().getOptions().setProperty(org.apache.axis2.transport.http.HTTPConstants.HEADER_ACCEPT_APPL_SOAP, "urn:ihe:iti:2007:CrossGatewayQuery");
			// turn chunking off
			queryStub._getServiceClient().getOptions().setProperty(org.apache.axis2.transport.http.HTTPConstants.CHUNKED, Boolean.FALSE);
			queryStub._getServiceClient().getOptions().setTimeOutInMilliSeconds(getQueryTimeout());

			ArrayList<org.apache.commons.httpclient.Header> customHttpHeaders = addTransactionContextFieldsToHeaders();
			// the HTTP header MUST be quoted because it contains special characters as defined in IETF RFC 822
			customHttpHeaders.add(new org.apache.commons.httpclient.Header("SOAPAction", "\"urn:ihe:iti:2007:CrossGatewayQuery\""));
			
			String patientId = filter.getPatientId();
	    	if(patientId != null && patientId.length() > 0)
	    		customHttpHeaders.add(new org.apache.commons.httpclient.Header(TransactionContextHttpHeaders.httpHeaderPatientId, 
	    				patientId));						
			
			queryStub._getServiceClient().getOptions().setProperty(
					org.apache.axis2.transport.http.HTTPConstants.HTTP_HEADERS, customHttpHeaders);
			
			//queryStub._getServiceClient().engageModule("addressing");
			
			//queryStub._getServiceClient().addStringHeader( new QName("http://www.w3.org/2005/08/addressing", "Action"), "urn:ihe:iti:2007:CrossGatewayQuery" );
//			try
//			{
//				OMNamespace omNamespace = OMAbstractFactory.getOMFactory().createOMNamespace("http://wso2.com", "ws");  
//				SOAPHeaderBlock secondHeader = OMAbstractFactory.getSOAP12Factory().createSOAPHeaderBlock("Action", omNamespace);  
//				secondHeader.addChild(AXIOMUtil.stringToOM("urn:ihe:iti:2007:CrossGatewayQuery"));
//				//secondHeader.setMustUnderstand(true);  
//				queryStub._getServiceClient().addHeader(secondHeader);
//			}
//			catch (XMLStreamException x)
//			{
//				x.printStackTrace();
//			}  
			
			org.oasis.ebxml.regrep.query.xmlbeans.AdhocQueryResponseDocument response = 
				queryStub.respondingGateway_CrossGatewayQuery(request);
			logger.info("CrossGatewayQuery completed execution, translating results");
			
			//dumpMessageContextProperties();

			DocumentSetResult result = 
				XCATranslatorXmlBeans.translate(
						response, 
						filter.getPatientId(),
						resolvedArtifactSource,
						getConfiguration().getProxyConfiguration().isAllowPartialSuccess().booleanValue()
					);
			transactionContext.addDebugInformation("DocumentSetResult: " + (result == null ? "null" : result.toString(true)));
			return result;
		}
		catch(RemoteException rX)
		{
			String msg = 
				"Error making XCA Query Request \n" +  
				"Binding name is " + queryStub._getServiceClient().getAxisService().getBindingName() + "'.";
			System.err.println(msg);
			logger.error( msg, rX);
			dumpMessageContextProperties();
			throw new MethodException(rX);
		}
		catch (TranslationException tX)
		{
			logger.error("Error translating XCA Query response", tX);
			throw new MethodException(tX);
		}
		
	}
	
	/**
	 * Verifies the provided URL contains a valid host and port to use 
	 * @param url
	 * @throws MethodException
	 */
	private void validateUrl(URL url)
	throws MethodException
	{
		if(url.getHost() == null || url.getHost().length() == 0)
			throw new MethodException("Cannot connect to url '" + url.toString() + "', missing host.");
		if(url.getPort() <= 0)
			throw new MethodException("Cannot connect to url '" + url.toString() + "', port is invalid.");
	}

	private void dumpMessageContextProperties()
	{
		try
		{
			Map<String, MessageContext> msgContexts = 
				queryStub._getServiceClient().getLastOperationContext().getMessageContexts();
			for(Map.Entry<String, MessageContext> msgContextEntry : msgContexts.entrySet())
			{
				MessageContext msgContext = msgContextEntry.getValue();
				logger.info("Dumping properties for " + flowText(msgContext.FLOW) + " message context '" + msgContext.toString() + "'.");
				for( Map.Entry<String, Object> propertyEntry : msgContext.getProperties().entrySet() )
					logger.info("\t'" + propertyEntry.getKey() + "' => '" + propertyEntry.getValue().toString() + "'.");
			}
		}
		catch(Exception x)
		{
			logger.error( "Unable to provide message details" );
		}
	}
	private String flowText(int flow)
	{
		switch(flow)
		{
		case MessageContext.IN_FLOW: return "in-flow";
		case MessageContext.OUT_FLOW: return "out-flow";
		case MessageContext.IN_FAULT_FLOW: return "in--fault-flow";
		case MessageContext.OUT_FAULT_FLOW: return "out-fault-flow";
		default: return "unknown flow direction";
		}
		
	}
	
	private ArrayList<org.apache.commons.httpclient.Header> addTransactionContextFieldsToHeaders()
	{
		ArrayList<org.apache.commons.httpclient.Header> customHttpHeaders = 
			new ArrayList<org.apache.commons.httpclient.Header>();
		
		// BHIE requires this header to know the request came from the VA			
		TransactionContext transactionContext = TransactionContextFactory.get();
		String sitename = transactionContext.getSiteName();
    	if(sitename != null && sitename.length() > 0)
    		customHttpHeaders.add(new org.apache.commons.httpclient.Header(TransactionContextHttpHeaders.httpHeaderSiteName, 
    				"VA_" + sitename));
    	
    	String fullname = transactionContext.getFullName();
    	if(fullname != null && fullname.length() > 0)
    		customHttpHeaders.add(new org.apache.commons.httpclient.Header(TransactionContextHttpHeaders.httpHeaderFullName, 
    				fullname));
    	
    	String sitenumber = transactionContext.getSiteNumber();
    	if(sitenumber != null && sitenumber.length() > 0)
    		customHttpHeaders.add(new org.apache.commons.httpclient.Header(TransactionContextHttpHeaders.httpHeaderSiteNumber, 
    				sitenumber));
    	
    	// for the user identifier, use the SSN or the user division and name or the transaction ID
    	String userIdentifier = 
    		transactionContext.getSsn() != null && transactionContext.getSsn().length() > 0 ?
    			transactionContext.getSsn() :
    			transactionContext.getLoggerUserDivision() != null && 
    			transactionContext.getLoggerUserDivision().length() > 0 && 
    			transactionContext.getLoggerFullName() != null &&
    			transactionContext.getLoggerFullName().length() > 0 ?
	    			transactionContext.getLoggerUserDivision() + ":" + transactionContext.getLoggerFullName() :
	    			transactionContext.getTransactionId() != null && transactionContext.getTransactionId().length() > 0 ? 
	    				transactionContext.getTransactionId() : null;
    	if(userIdentifier != null && userIdentifier.length() > 0)
    		customHttpHeaders.add(
    			new org.apache.commons.httpclient.Header(TransactionContextHttpHeaders.httpHeaderSSN, userIdentifier)
    		);
    	
    	customHttpHeaders.add(new org.apache.commons.httpclient.Header(TransactionContextHttpHeaders.httpHeaderPurposeOfUse, 
				Requestor.PurposeOfUse.routineMedicalCare.getDescription()));
    	
    	String transactionId = transactionContext.getTransactionId();
    	if(transactionId != null && transactionId.length() > 0)
    		customHttpHeaders.add(new org.apache.commons.httpclient.Header(TransactionContextHttpHeaders.httpHeaderTransactionId, 
    				transactionId));    	    
    	
    	return customHttpHeaders;
	}
	
	/**
	 * 
	 * @param documentUrn
	 * @return
	 * @throws MethodException
	 * @throws ConnectionException
	 */
	/*
	public ImageStreamResponse getPatientDocument(DocumentURN documentUrn)
	throws MethodException, ConnectionException
	{
		org.ihe.iti.xdsb.adb.RetrieveDocumentSetRequest request = null;
		
		logger.info("Creating XCA Retrieve document request for document '" + documentUrn.toString() + "'.");
		request = this.adbTranslator.createRetrieveDocumentSetRequest(documentUrn);

		try
		{
			logger.info("Executing CrossGatewayRetrieve to url '" + retrieveUrl + "'");
			org.ihe.iti.xdsb.adb.RetrieveDocumentSetResponse response = 
				retrieveStub.respondingGateway_CrossGatewayRetrieve(request);
			logger.info("CrossGatewayRequest completed execution, translating results");

			return this.adbTranslator.translateDocumentRetrieveResponse(response);

		}
		catch(IOException ioe)
		{
			logger.error("Error making XCA Retrieve", ioe);
			throw new MethodException(ioe);			
		}
	}*/

	/**
	 * @param homeCommunityUid
	 * @param repositoryUniqueId
	 * @param documentId
	 * @return
	 * @throws MethodException 
	 */
	public ImageStreamResponse getPatientDocument(String homeCommunityUid, String repositoryUniqueId, String documentId) 
	throws MethodException
	{
		org.ihe.iti.xdsb.adb.RetrieveDocumentSetRequest request = null;
		validateUrl(getRetrieveUrl());
		
		logger.info("Creating XCA Retrieve document request for document '" + homeCommunityUid + ":" + repositoryUniqueId + ":" + documentId + "'.");
		request = this.adbTranslator.createRetrieveDocumentSetRequest(homeCommunityUid, repositoryUniqueId, documentId);
		
		// add transaction context fields necessary to make 
		ArrayList<org.apache.commons.httpclient.Header> customHttpHeaders = addTransactionContextFieldsToHeaders();
		Options clientOptions = retrieveStub._getServiceClient().getOptions();
		clientOptions.setProperty( org.apache.axis2.transport.http.HTTPConstants.HTTP_HEADERS, customHttpHeaders );
		//clientOptions.setProperty(Constants.Configuration.ENABLE_MTOM, Constants.VALUE_TRUE);
	        
		try
		{
			logger.info("Executing CrossGatewayRetrieve to url '" + retrieveUrl + "'");
			TransactionContext transactionContext = TransactionContextFactory.get();
			transactionContext.addDebugInformation("Executing CrossGatewayRetrieve to url '" + retrieveUrl + "'");
			org.ihe.iti.xdsb.adb.RetrieveDocumentSetResponse response = 
				retrieveStub.respondingGateway_CrossGatewayRetrieve(request);
			logger.info("CrossGatewayRequest completed execution, translating results");

			return this.adbTranslator.translate(response);

		}
		catch(IOException ioe)
		{
			logger.error("Error making XCA Retrieve", ioe);
			throw new MethodException(ioe);			
		}
	}

}
